bl_info = {
    "name": "Blendit",
    "description": "Creates dynamic blends of objects",
    "author": "Borek Bures (Remesher)",
    "version": (1, 5, 0),
    "blender": (2, 83, 0),
    "location": "View3D > Slidebar > Blendit",
    "warning": "",
    "doc_url": "https://gum.co/PlPsJ",
    "tracker_url": "https://blenderartists.org/t/blendit-full-dynamic-material-blending-of-objects/1253712",
    "category": "Object"}


import bpy
from bpy.utils import (register_class,
                       unregister_class
                       )
from bpy.props import (BoolProperty,
                       IntProperty,
                       FloatProperty,
                       
                       EnumProperty,
                       PointerProperty,
                       
                       StringProperty,
                       )
from bpy.types import PropertyGroup
from mathutils import Vector


class BlenditObjectProperties(PropertyGroup):

    blend_min_dist: FloatProperty(
        name = "Min Distance",
        description = "Blending start distance",
        subtype = "DISTANCE",
        default = 0.05,
        min = 0.0,
        soft_max = 1.0
        )

    blend_dist: FloatProperty(
        name = "Max Distance",
        description = "Maximum blending range",
        subtype = "DISTANCE",
        default = 0.5,
        min = 0.00001,
        soft_max = 1.0
        )

    blend_scale: FloatProperty(
        name = "Individual Scale",
        description = "Scale of the blending range of the individual object",
        subtype = "FACTOR",
        default = 1,
        min = 0.00001,
        soft_max = 2.0
        )

    source_obj: PointerProperty(
        name = "Source Object",
        description = "",
        type=bpy.types.Object
        )

    blended: BoolProperty(
        name = "Blended",
        default = False
        )

    blend_source: BoolProperty(
        name = "Blend Source",
        default = False
        )

    blend_normals: BoolProperty( 
        name = "Blend Normals",
        description = "Created blend of normals",
        default = True
        )

    blended_normals: BoolProperty( 
        name = "Blended Normals",
        default = False
        )

    reused_uv: BoolProperty( 
        name = "Reused UV",
        default = False
        )

    advanced_mod: BoolProperty(
        name = "Advanced Mod",
        default = False
        )

    blend_normal_factor: FloatProperty(
        name = "Blend Normals Factor",
        description = "Blend of normals factor",
        subtype = "FACTOR",
        default = 1.0,
        min = 0.0,
        max = 1.0
        )
    
    applied: BoolProperty(
        name = "Applied Dynamic Blend",
        default = False
        )

    auto_smooth_changed: BoolProperty(
        name = "Auto Smooth Changed",
        description = "Auto Smooth changed",
        default = False
        )

    blend_style: IntProperty(
        name = "Blend Style",
        description = "Blend gradient style",
        default = 0
        )

    style_preview: BoolProperty(
        name = "Mask Preview",
        description = "Show black and white mask",
        default = False
        )

    style_preview_levels: BoolProperty(
        name = "Levels Preview",
        description = "Show min/max values",
        default = False
        )

    style_seed: IntProperty(
        name = "Seed",
        description = "Noise seed",
        default = 0
        )

    style_brightness: FloatProperty(
        name = "Threshold",
        description = "Additional level",
        subtype = "FACTOR",
        default = 0,
        soft_min = -10.0,
        soft_max = 10.0
        )

    style_brightness2: FloatProperty(
        name = "Threshold 2",
        description = "Additional level",
        subtype = "FACTOR",
        default = 0,
        soft_min = -10.0,
        soft_max = 10.0
        )

    style_amplitude: FloatProperty(
        name = "Amplitude",
        description = "Amplitude",
        subtype = "FACTOR",
        default = 5.0,
        soft_min = -100.0,
        soft_max = 100.0
        )

    style_scale: FloatProperty(
        name = "Scale",
        description = "Scale",
        subtype = "FACTOR",
        default = 1.0,
        soft_min = -100.0,
        soft_max = 100.0
        )

    style_scale2: FloatProperty(
        name = "Scale 2",
        description = "Scale",
        subtype = "FACTOR",
        default = 1.0,
        soft_min = -100.0,
        soft_max = 100.0
        )

    style_frequency: FloatProperty(
        name = "Frequency",
        description = "Frequency",
        subtype = "FACTOR",
        default = 5.0,
        min = 0.0001,
        soft_max = 100.0
        )

    style_detail: FloatProperty(
        name = "Detail",
        description = "Detail",
        subtype = "FACTOR",
        default = 5.0,
        min = 0.0,
        max = 16.0
        )

    style_roughness: FloatProperty(
        name = "Roughness",
        description = "Roughness",
        subtype = "FACTOR",
        default = 0.5,
        min = 0.0,
        max = 1.0
        )

    style_distortion: FloatProperty(
        name = "Distortion",
        description = "Distortion",
        subtype = "FACTOR",
        default = 0,
        soft_min = -10.0,
        soft_max = 10.0
        )

    style_smooth: FloatProperty(
        name = "Radius",
        description = "Factor of smooth blend",
        subtype = "FACTOR",
        default = 0.75,
        min = 0.0,
        max = 1.0
        )

    style_reroute: StringProperty(
        name = "Reroute Label",
        description = "The output defined by reroute node whitch will be automatic connected",
        default = "height_new"
        )

    uv_scale: FloatProperty(
        name = "Scale",
        description = "UV Scale",
        subtype = "FACTOR",
        default = 1.0,
        soft_min = 0.0,
        soft_max = 2.0
        )


class BlenditMaterialProperties(PropertyGroup):

    style_reroute: StringProperty(
        name = "Reroute Label",
        description = "The data output(s) for blend style input(s) defined by reroute node",
        default = "height"
        )


class BlenditSceneProperties(PropertyGroup):

    new_mat: BoolProperty(
        name = "Create Material",
        description = "Create blend of two materials",
        default = True
        )

    transfer_uv: BoolProperty(
        name = "Transfer UV",
        description = "Transfer UV from source object",
        default = True
        )

    blend_normals: BoolProperty(
        name = "Blend Normals",
        description = "Create blend of normals",
        default = True
        )

    auto_smooth: BoolProperty(
        name = "Auto Smooth",
        description = "Blend of normals using Auto Smooth",
        default = True
        )


BLENDIT_ID_NAME = "Blendit"
BLENDIT_BLEND_ID_NAME = "Blendit_blend"
BLENDIT_NODEGROUP_MIX = BLENDIT_ID_NAME + "_Mix_Shader"
BLENDIT_NODEGROUP_CUSTOM = BLENDIT_ID_NAME + "_Custom_Blend"
BLENDIT_NODEGROUP_NOISE = BLENDIT_ID_NAME + "_Noise_Blend"
BLENDIT_NODEGROUP_HEIGHT = BLENDIT_ID_NAME + "_Heights_Blend"

class Blendit_OT_Blendit(bpy.types.Operator):
    bl_label = "Create Dynamic Blends"
    bl_idname = "blendit.blendit"
    bl_description = "Create dynamic material blend on the selected objects, using the active object as the source"

    action: StringProperty(
        name = "",
        default = ""
        )

    @classmethod
    def poll(cls, context):
        scene = context.scene
        p = (context.mode == 'OBJECT' and
            isinstance(context.active_object, bpy.types.Object) and isinstance(context.active_object.data, bpy.types.Mesh)
            
            )
        return p


    def execute(self, context):

        def create_vertex_group(_obj, _name):
            if not _name in _obj.vertex_groups:
                _v_group = _obj.vertex_groups.new(name=_name)
            else:
                _v_group = _obj.vertex_groups[_name]
            for _indx, _vert in enumerate(_obj.data.vertices):
                _v_group.add([_indx], 1.0, 'ADD')
            return

        def create_vertex_color(_obj, _name, _color):
            if not _name in _obj.data.vertex_colors:
                _obj.data.vertex_colors.new(name=_name, do_init=False)
            if _color == 0:
                for _vert in _obj.data.vertex_colors[_name].data:
                    _vert.color = (0.0, 0.0, 0.0, 1.0)
            return

        def duplicate_mapping(_obj, _source_name, _target_name): 
            if not _target_name in _obj.data.uv_layers: 
                _obj.data.uv_layers[_source_name].active = True
                _new_uv = _obj.data.uv_layers.new(name=_target_name)
                _new_uv.active = True
            else: 
                _new_uv = _obj.data.uv_layers[_target_name]
                _new_uv.active = True
            return _new_uv

        def create_mapping(_obj, _name):
            if not _name in _obj.data.uv_layers:
                _new_uv = _obj.data.uv_layers.new(name=_name)
                _new_uv.active = True
            return

        def create_modifiers(_obj, _name):

            def get_compatible_param(modifier, param):
                if param == "layers_vcol_select_src":
                    if bpy.app.version >= (3, 2, 0):
                        _param = modifier.layers_vcol_loop_select_src
                    else:
                        _param = modifier.layers_vcol_select_src
                elif param == "layers_vcol_select_dst":
                    if bpy.app.version >= (3, 2, 0):
                        _param = modifier.layers_vcol_loop_select_dst
                    else:
                        _param = modifier.layers_vcol_select_dst
                return _param

            _mod_name = _name + "_V_W_Proximity"
            if _mod_name in _obj.modifiers:
                _obj.modifiers.remove(_obj.modifiers[_mod_name])
            _obj.modifiers.new(name=_mod_name, type='VERTEX_WEIGHT_PROXIMITY')
            _obj.modifiers[_mod_name].vertex_group = _name
            _obj.modifiers[_mod_name].target = active_obj
            _obj.modifiers[_mod_name].proximity_mode = 'GEOMETRY'
            _obj.modifiers[_mod_name].proximity_geometry = {'FACE'}
            _obj.modifiers[_mod_name].min_dist = active_obj.blendit_locals.blend_dist 
            _obj.modifiers[_mod_name].max_dist = 0
            _obj.modifiers[_mod_name].falloff_type = 'SHARP'
            _obj.modifiers[_mod_name].show_expanded = False

            _mod_name = _name + "_D_Transfer_Vcol"
            if _mod_name in _obj.modifiers:
                _obj.modifiers.remove(_obj.modifiers[_mod_name])
            _obj.modifiers.new(name=_mod_name, type='DATA_TRANSFER')
            _obj.modifiers[_mod_name].object = active_obj
            _obj.modifiers[_mod_name].use_object_transform = False 
            _obj.modifiers[_mod_name].use_loop_data = True
            _obj.modifiers[_mod_name].loop_mapping = 'NEAREST_NORMAL' 
            _obj.modifiers[_mod_name].data_types_loops = {'VCOL'}
            _param = get_compatible_param(_obj.modifiers[_mod_name], 'layers_vcol_select_src')
            _param = BLENDIT_BLEND_ID_NAME
            _param = get_compatible_param(_obj.modifiers[_mod_name], 'layers_vcol_select_dst')
            _param = 'NAME'
            _obj.modifiers[_mod_name].vertex_group = _name
            _obj.modifiers[_mod_name].show_expanded = False

            _mod_name = _name + "_D_Transfer_Norm"
            if _mod_name in _obj.modifiers:
                _obj.modifiers.remove(_obj.modifiers[_mod_name])
            if blendit_globals.blend_normals:
                _obj.modifiers.new(name=_mod_name, type='DATA_TRANSFER')
                _obj.modifiers[_mod_name].object = active_obj
                _obj.modifiers[_mod_name].use_loop_data = True
                _obj.modifiers[_mod_name].loop_mapping = 'POLYINTERP_NEAREST' 
                _obj.modifiers[_mod_name].data_types_loops = {'CUSTOM_NORMAL'}
                _obj.modifiers[_mod_name].vertex_group = _name
                _obj.modifiers[_mod_name].show_expanded = False
                _obj.blendit_locals.blend_normals = True

                if blendit_globals.auto_smooth: 
                    
                    if not _obj.data.use_auto_smooth:
                        _obj.data.use_auto_smooth = True
                        _obj.data.auto_smooth_angle = 3.14159
                        _obj.blendit_locals.auto_smooth_changed = True
            else:
                _obj.blendit_locals.blend_normals = False

            _mod_name = _name + "_D_Transfer_UV"
            if _mod_name in _obj.modifiers:
                _obj.modifiers.remove(_obj.modifiers[_mod_name])
            if blendit_globals.transfer_uv and active_uv:
                _obj.modifiers.new(name=_mod_name, type='DATA_TRANSFER')
                _obj.modifiers[_mod_name].object = active_obj
                _obj.modifiers[_mod_name].use_loop_data = True
                if blendit_globals.blend_normals:
                    _obj.modifiers[_mod_name].loop_mapping = 'POLYINTERP_LNORPROJ'
                else:
                    _obj.modifiers[_mod_name].loop_mapping = 'POLYINTERP_NEAREST'
                _obj.modifiers[_mod_name].data_types_loops = {'UV'}
                _obj.modifiers[_mod_name].layers_uv_select_src = active_uv 
                _obj.modifiers[_mod_name].show_expanded = False

            _mod_name = _name + "_UV_Warp"
            if _mod_name in _obj.modifiers:
                _obj.modifiers.remove(_obj.modifiers[_mod_name])
            if not blendit_globals.transfer_uv and active_uv:
                _obj.modifiers.new(name=_mod_name, type='UV_WARP')
                _obj.modifiers[_mod_name].uv_layer = active_uv
                _obj.modifiers[_mod_name].show_in_editmode = False
                _obj.modifiers[_mod_name].show_expanded = False
                _obj.blendit_locals.reused_uv = True

            return

        def create_drivers(_obj_source, _obj_target):
            if _obj_target.animation_data and _obj_target.animation_data.drivers: 
                for _driver in _obj_target.animation_data.drivers:
                    if _driver.data_path == 'modifiers["'+BLENDIT_ID_NAME+'_V_W_Proximity"].min_dist':
                        _obj_target.modifiers[BLENDIT_ID_NAME+"_V_W_Proximity"].driver_remove("min_dist")

                    if _driver.data_path == 'modifiers["'+BLENDIT_ID_NAME+'_V_W_Proximity"].max_dist':
                        _obj_target.modifiers[BLENDIT_ID_NAME+"_V_W_Proximity"].driver_remove("max_dist")

                    if _driver.data_path == 'modifiers["'+BLENDIT_ID_NAME+'_D_Transfer_Norm"].mix_factor':
                        _obj_target.modifiers[BLENDIT_ID_NAME+"_D_Transfer_Norm"].driver_remove("mix_factor")
                        _obj_target.modifiers[BLENDIT_ID_NAME+"_D_Transfer_Norm"].mix_factor = 1

                    if _driver.data_path == 'modifiers["'+BLENDIT_ID_NAME+'_UV_Warp"].scale':
                        _obj_target.modifiers[BLENDIT_ID_NAME+"_UV_Warp"].driver_remove("scale")

            if BLENDIT_ID_NAME+"_V_W_Proximity" in _obj_target.modifiers: 
                
                _driver = _obj_target.modifiers[BLENDIT_ID_NAME+"_V_W_Proximity"].driver_add("min_dist").driver

                _var1 = _driver.variables.new()
                _var1.name = "Blendit_blend_dist"
                _var1.targets[0].id = _obj_source
                _var1.targets[0].data_path = 'blendit_locals.blend_dist'

                _var2 = _driver.variables.new()
                _var2.name = "Blendit_blend_scale"
                _var2.targets[0].id = _obj_target
                _var2.targets[0].data_path = 'blendit_locals.blend_scale'

                _driver.expression = _var1.name + "*" + _var2.name

                _driver = _obj_target.modifiers[BLENDIT_ID_NAME+"_V_W_Proximity"].driver_add("max_dist").driver

                _var1 = _driver.variables.new()
                _var1.name = "Blendit_blend_dist"
                _var1.targets[0].id = _obj_source
                if _obj_target.blendit_locals.advanced_mod:
                    _var1.targets[0].data_path = 'blendit_locals.blend_min_dist'            
                else:
                    _var1.targets[0].data_path = 'blendit_locals.blend_dist'

                _var2 = _driver.variables.new()
                _var2.name = "Blendit_blend_scale"
                _var2.targets[0].id = _obj_target
                _var2.targets[0].data_path = 'blendit_locals.blend_scale'

                if _obj_target.blendit_locals.advanced_mod:
                    _driver.expression = _var1.name + "*" + _var2.name
                else:
                    _driver.expression = _var1.name + "*" + _var2.name + "*0.1"

            if _obj_target.blendit_locals.advanced_mod and _obj_target.blendit_locals.blend_normals and BLENDIT_ID_NAME+"_D_Transfer_Norm" in _obj_target.modifiers: 
                _driver = _obj_target.modifiers[BLENDIT_ID_NAME+"_D_Transfer_Norm"].driver_add("mix_factor").driver

                _var1 = _driver.variables.new()
                _var1.name = "Blendit_normal_fact"
                _var1.targets[0].id = _obj_target
                _var1.targets[0].data_path = 'blendit_locals.blend_normal_factor'
                
                _driver.expression = _var1.name

            if BLENDIT_ID_NAME + "_UV_Warp" in _obj_target.modifiers: 
                _drivers = _obj_target.modifiers[BLENDIT_ID_NAME + "_UV_Warp"].driver_add("scale")

                _var1 = _drivers[0].driver.variables.new()
                _var1.name = "Blendit_uv_scale_u"
                _var1.targets[0].id = _obj_target
                _var1.targets[0].data_path = 'blendit_locals.uv_scale'
                
                _drivers[0].driver.expression = _var1.name

                _var1 = _drivers[1].driver.variables.new()
                _var1.name = "Blendit_uv_scale_v"
                _var1.targets[0].id = _obj_target
                _var1.targets[0].data_path = 'blendit_locals.uv_scale'
                
                _drivers[1].driver.expression = _var1.name            

            return

        
        def copy_node_attributes(from_prop, to_prop, attributes = None, to_nodetree = None):
            ignored_attributes = ("rna_type", "type", "dimensions", "inputs", "outputs", "internal_links", "select",
                                  "texture_mapping", "color_mapping", "image_user", "elements", "mapping", "color_ramp", "curves", "points") 
            
            if attributes:
                for _attr in attributes:  
                    if hasattr(to_prop, _attr): 
                        try:
                            setattr(to_prop, _attr, getattr(from_prop, _attr))
                        except:
                            pass
            else:    
                for _attr in from_prop.bl_rna.properties:
                    if not _attr.identifier in ignored_attributes and not _attr.identifier.split("_")[0] == "bl":
                        if hasattr(to_prop, _attr.identifier): 
                            try:
                                if _attr.identifier == "parent": 
                                    setattr(to_prop, _attr.identifier, to_nodetree[getattr(from_prop, _attr.identifier).name]) 
                                else:
                                    setattr(to_prop, _attr.identifier, getattr(from_prop, _attr.identifier))
                            except:
                                pass
            return

        def copy_node(source_node, to_nodes):
            _input_attributes = ("default_value", "name") 
            _output_attributes = ("default_value", "name")

            _new_node = to_nodes.new(source_node.bl_idname) 
            
            copy_node_attributes(source_node, _new_node, to_nodetree=to_nodes)

            for _indx, _inpt in enumerate(source_node.inputs): 
                
                copy_node_attributes(_inpt, _new_node.inputs[_indx], _input_attributes)

            for _indx, _oupt in enumerate(source_node.outputs): 
                
                copy_node_attributes(_oupt, _new_node.outputs[_indx], _output_attributes)

            if source_node.bl_idname == "ShaderNodeValToRGB":
                copy_node_attributes(source_node.color_ramp, _new_node.color_ramp) 
                _new_node.color_ramp.elements.remove(_new_node.color_ramp.elements[1]) 
                for indx, _element in enumerate(source_node.color_ramp.elements): 
                    if indx != 0:
                        _new_node.color_ramp.elements.new(0)
                for indx, _element in enumerate(source_node.color_ramp.elements): 
                    copy_node_attributes(_element, _new_node.color_ramp.elements[indx])
                
            if (source_node.bl_idname == "ShaderNodeRGBCurve"
                or source_node.bl_idname == "ShaderNodeVectorCurve"
                or source_node.bl_idname == "ShaderNodeFloatCurve"):
                copy_node_attributes(source_node.mapping, _new_node.mapping) 

                for indx, _curve in enumerate(source_node.mapping.curves): 
                    copy_node_attributes(source_node.mapping.curves[indx], _new_node.mapping.curves[indx]) 

                    for indx2, _point in enumerate(_curve.points): 
                        if indx2 != 0 and indx2 != len(_curve.points)-1: 
                            _new_node.mapping.curves[indx].points.new(0, 0)

                    for indx2, _point in enumerate(_curve.points): 
                        copy_node_attributes(_point, _new_node.mapping.curves[indx].points[indx2])

                _new_node.mapping.update()
                
            return _new_node

        def copy_nodes_links(from_nodes, to_node_tree, use_uv): 
            for _from_node in from_nodes:
                
                _target_node = to_node_tree.nodes[_from_node.name]

                for _indx, _input in enumerate(_from_node.inputs):
                    for _link in _input.links:
                        
                        _connected_node = to_node_tree.nodes[_link.from_node.name]
                        
                        if use_uv and _connected_node.bl_idname == "ShaderNodeTexCoord" and _link.from_socket.name == "UV":
                            _new_node = to_node_tree.nodes.new("ShaderNodeUVMap")
                            _new_node.location = Vector((_connected_node.location[0], _connected_node.location[1] + 150))
                            _new_node.uv_map = use_uv
                            to_node_tree.links.new(_new_node.outputs[0],_target_node.inputs[_indx])
                        else: 
                            to_node_tree.links.new(_connected_node.outputs[_link.from_socket.name], _target_node.inputs[_indx])

            return
        
        def connect_unconnected_nodes(from_nodes, in_node_tree, use_uv): 
            _input_name = "Vector" 
            _new_node = None
            _in_nodes = in_node_tree.nodes

            if use_uv:
                for _from_node in from_nodes:
                    if _from_node.bl_idname == "ShaderNodeTexImage": 
                        if not _in_nodes[_from_node.name].inputs[_input_name].is_linked: 
                            if not _new_node: 
                                _new_node = in_node_tree.nodes.new("ShaderNodeUVMap")
                                _new_node.location = Vector((9999999, 0))
                                _new_node.uv_map = use_uv

                            _location = get_node_location(_in_nodes[_from_node.name])
                            if _new_node.location[0] > _location[0] - 250: 
                                _new_node.location = Vector((_location[0] - 250, _location[1])) 
                            in_node_tree.links.new(_new_node.outputs[0], _in_nodes[_from_node.name].inputs[_input_name]) 
            return

        def check_material_nodes(from_nodes):
            _warning = 0
            unsupported_nodes = ("ShaderNodeTangent", "ShaderNodeVertexColor",
                                 "ShaderNodeAttribute", "ShaderNodeHairInfo", "ShaderNodeObjectInfo", "ShaderNodeParticleInfo", "ShaderNodeVolumeInfo", "ShaderNodeWireFrame")

            for _from_node in from_nodes: 
                if _from_node.bl_idname in unsupported_nodes:
                    for _output in _from_node.outputs: 
                        if _output.is_linked:
                            if not (_warning & 0b010): 
                                _warning += 0b010
                            
                if ((_from_node.bl_idname == "ShaderNodeTexCoord" and _from_node.outputs[0].is_linked) or 
                    (_from_node.bl_idname == "ShaderNodeTexCoord" and _from_node.outputs[2].is_linked and active_uv == None) or 
                    ((_from_node.bl_idname == "ShaderNodeUVMap" and _from_node.uv_map != "" and _from_node.outputs[0].is_linked) and 
                    (_from_node.bl_idname == "ShaderNodeUVMap" and _from_node.uv_map != active_render_uv and _from_node.outputs[0].is_linked)) or 
                    (_from_node.bl_idname == "ShaderNodeUVMap" and active_uv == None and _from_node.outputs[0].is_linked)): 
                    if not (_warning & 0b100): 
                        _warning += 0b100
                    
            return _warning
        
        def get_node_location(node):
            location = Vector((0.0, 0.0))
            if node.parent:
                location = get_node_location(node.parent)
            location = location + node.location
            return location

        def get_nodes_borders(for_nodes):
            _min_loc = Vector((9999999, 9999999))
            _max_loc = Vector((-9999999, -9999999))
            
            for _node in for_nodes:
                location = get_node_location(_node)
                if location[0] < _min_loc[0]:
                    _min_loc[0] = location[0]
                if (location[0] + _node.width) > _max_loc[0]:
                    _max_loc[0] = location[0] + _node.width

                if location[1] < _min_loc[1]:
                    _min_loc[1] = location[1]
                if (location[1] + _node.height) > _max_loc[1]:
                    _max_loc[1] = location[1] + _node.height

            return _min_loc, _max_loc

        def create_nodegroup_mixshader(for_nodes): 

            if not BLENDIT_NODEGROUP_MIX in bpy.data.node_groups: 
                _nodegroup = bpy.data.node_groups.new(name=BLENDIT_NODEGROUP_MIX, type="ShaderNodeTree")

                _nodegroup_input = _nodegroup.nodes.new("NodeGroupInput") 
                _nodegroup.inputs.new('NodeSocketFloatFactor', "Gradient")
                _nodegroup.inputs.new('NodeSocketShader', "Object Shader")
                _nodegroup.inputs.new('NodeSocketShader', "Source Shader")
                _nodegroup_input.location = Vector((-300, 50))

                _nodegroup_output = _nodegroup.nodes.new("NodeGroupOutput") 
                _nodegroup.outputs.new('NodeSocketShader', "Shader")
                _nodegroup.outputs.new('NodeSocketFloatFactor', "Gradient")
                _nodegroup.outputs.new('NodeSocketColor', "Levels")
                _nodegroup_output.location = Vector((200, 50))
                
                _nodegroup_mix = _nodegroup.nodes.new("ShaderNodeMixShader") 
                _nodegroup_mix.location = Vector((-50, 150))
                
                _nodegroup_cor = _nodegroup.nodes.new("ShaderNodeValToRGB") 
                _nodegroup_cor.color_ramp.elements.new(0.001)
                _nodegroup_cor.color_ramp.elements.new(0.999)
                _nodegroup_cor.color_ramp.elements[0].color = (0.0, 1.0, 0.0, 1.0)
                _nodegroup_cor.color_ramp.elements[1].color = (0.0, 0.0, 0.0, 1.0)
                _nodegroup_cor.color_ramp.elements[2].color = (1.0, 1.0, 1.0, 1.0)
                _nodegroup_cor.color_ramp.elements[3].color = (1.0, 0.0, 0.0, 1.0)
                _nodegroup_cor.location = Vector((-100, -50))

                _nodegroup.links.new(_nodegroup_input.outputs[0], _nodegroup_mix.inputs[0]) 
                _nodegroup.links.new(_nodegroup_input.outputs[1], _nodegroup_mix.inputs[1])
                _nodegroup.links.new(_nodegroup_input.outputs[2], _nodegroup_mix.inputs[2])
                _nodegroup.links.new(_nodegroup_input.outputs[0], _nodegroup_output.inputs[1])
                _nodegroup.links.new(_nodegroup_input.outputs[0], _nodegroup_cor.inputs[0])

                _nodegroup.links.new(_nodegroup_mix.outputs[0], _nodegroup_output.inputs[0])
                _nodegroup.links.new(_nodegroup_cor.outputs[0], _nodegroup_output.inputs[2])

            _nodegroup = for_nodes.new('ShaderNodeGroup') 
            _nodegroup.node_tree = bpy.data.node_groups[BLENDIT_NODEGROUP_MIX]
            _nodegroup.name = BLENDIT_NODEGROUP_MIX 
            return _nodegroup

        def create_nodegroup_custom(for_nodes): 

            if BLENDIT_NODEGROUP_CUSTOM in bpy.data.node_groups:
                bpy.data.node_groups.remove(bpy.data.node_groups[BLENDIT_NODEGROUP_CUSTOM]) 

            if not BLENDIT_NODEGROUP_CUSTOM in bpy.data.node_groups: 
                _nodegroup = bpy.data.node_groups.new(name=BLENDIT_NODEGROUP_CUSTOM, type="ShaderNodeTree")

                _nodegroup_input = _nodegroup.nodes.new("NodeGroupInput") 
                _input =_nodegroup.inputs.new('NodeSocketFloatFactor', "Weight")
                _input.default_value = 0.5
                _nodegroup.inputs.new('NodeSocketFloat', "Custom")
                _input = _nodegroup.inputs.new('NodeSocketFloatFactor', "Amplitude")
                _input.min_value = -100.0
                _input.max_value = 100.0
                _input.default_value = 1.0
                _input = _nodegroup.inputs.new('NodeSocketFloatFactor', "Threshold")
                _input.min_value = -10.0
                _input.max_value = 10.0
                _input = _nodegroup.inputs.new('NodeSocketFloatFactor', "Radius")
                _input.min_value = 0
                _input.max_value = 1.0
                _input.default_value = 1.0
                _nodegroup_input.location = Vector((-1800, 50))

                _nodegroup_output = _nodegroup.nodes.new("NodeGroupOutput") 
                _output = _nodegroup.outputs.new('NodeSocketFloatFactor', "Fac")
                _nodegroup_output.location = Vector((1700, 50))

                _nodegroup_node_1 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_1.operation = 'DIVIDE'
                _nodegroup_node_1.inputs[1].default_value = 2.0
                _nodegroup.links.new(_nodegroup_input.outputs[2], _nodegroup_node_1.inputs[0])
                _nodegroup_node_1.location = Vector((-1400, 85))

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_2.operation = 'SUBTRACT'
                _nodegroup_node_2.inputs[0].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_node_2.inputs[1])
                _nodegroup_node_2.location = Vector((-1200, 150))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'ADD'
                _nodegroup_node_3.inputs[1].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_node_3.inputs[0])
                _nodegroup_node_3.location = Vector((-1200, -25))

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMapRange") 
                _nodegroup_node_4.clamp = False
                _nodegroup_node_4.inputs[1].default_value = 0.0
                _nodegroup_node_4.inputs[2].default_value = 1.0
                _nodegroup.links.new(_nodegroup_input.outputs[1], _nodegroup_node_4.inputs[0])
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_4.inputs[3])
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_4.inputs[4])
                _nodegroup_node_4.location = Vector((-1000, 110))

                _nodegroup_node_1 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_1.operation = 'ADD'
                _nodegroup_node_1.use_clamp = True
                _nodegroup_node_1.inputs[1].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_1.inputs[0])
                _nodegroup.links.new(_nodegroup_input.outputs[3], _nodegroup_node_1.inputs[1])
                _nodegroup_node_1.location = Vector((-800, 0))

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_2.operation = 'ADD'
                _nodegroup_node_2.use_clamp = True
                _nodegroup_node_2.inputs[1].default_value = 0
                _nodegroup.links.new(_nodegroup_input.outputs[0], _nodegroup_node_2.inputs[0])
                _nodegroup_node_2.location = Vector((-1000, 300))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'SUBTRACT'
                _nodegroup_node_3.inputs[0].default_value = 1.0
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_3.inputs[1])
                _nodegroup_node_3.location = Vector((-500, 450))

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_4.operation = 'SUBTRACT'
                _nodegroup_node_4.inputs[0].default_value = 1.0
                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_node_4.inputs[1])
                _nodegroup_node_4.location = Vector((-500, 250))                

                _nodegroup_node_5 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_5.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_5.inputs[0])
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_5.inputs[1])
                _nodegroup_node_5.location = Vector((-300, 400))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_node_5.outputs[0], _nodegroup_node_3.inputs[0])
                _nodegroup_node_3.inputs[1].default_value = 2.0
                _nodegroup_node_3.location = Vector((-100, 400))

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_4.operation = 'SUBTRACT'
                _nodegroup_node_4.inputs[0].default_value = 1.0
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_4.inputs[1])
                _nodegroup_node_4.location = Vector((100, 400))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_3.inputs[0])
                _nodegroup_node_3.location = Vector((300, 300))

                
                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_4.operation = 'GREATER_THAN'
                _nodegroup_node_4.use_clamp = True
                _nodegroup_node_4.inputs[1].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_4.inputs[0])
                _nodegroup_node_4.location = Vector((-100, 200))

                _nodegroup_node_5 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_5.operation = 'COMPARE'
                _nodegroup_node_5.use_clamp = True
                _nodegroup_node_5.inputs[1].default_value = 0.5
                _nodegroup_node_5.inputs[2].default_value = 0
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_5.inputs[0])
                _nodegroup_node_5.location = Vector((-100, 0))

                _nodegroup_node_6 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_6.operation = 'ADD'
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_6.inputs[0])
                _nodegroup.links.new(_nodegroup_node_5.outputs[0], _nodegroup_node_6.inputs[1])
                _nodegroup.links.new(_nodegroup_node_6.outputs[0], _nodegroup_node_3.inputs[1])
                _nodegroup_node_6.location = Vector((100, 150))
                
                
                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_4.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_4.inputs[0])
                _nodegroup_node_4.inputs[1].default_value = 2.0
                _nodegroup_node_4.location = Vector((-100, -200))
                
                _nodegroup_node_5 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_5.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_5.inputs[0])
                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_node_5.inputs[1])
                _nodegroup_node_5.location = Vector((100, -250))

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_4.operation = 'LESS_THAN'
                _nodegroup_node_4.use_clamp = True
                _nodegroup_node_4.inputs[1].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_4.inputs[0])
                _nodegroup_node_4.location = Vector((100, -450))

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_2.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_node_5.outputs[0], _nodegroup_node_2.inputs[0])
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_2.inputs[1])
                _nodegroup_node_2.location = Vector((300, -250))

                _nodegroup_node_1 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_1.operation = 'ADD'
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_1.inputs[0])
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_1.inputs[1])
                _nodegroup_node_1.location = Vector((500, 50))

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMapRange") 
                _nodegroup_node_2.clamp = False
                _nodegroup_node_2.inputs[1].default_value = 0.0
                _nodegroup_node_2.inputs[2].default_value = 1.0
                _nodegroup_node_2.inputs[3].default_value = 0.01
                _nodegroup_node_2.inputs[4].default_value = 0.0001
                _nodegroup.links.new(_nodegroup_input.outputs[4], _nodegroup_node_2.inputs[0])
                _nodegroup_node_2.location = Vector((300, -700))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'POWER'
                _nodegroup.links.new(_nodegroup_input.outputs[4], _nodegroup_node_3.inputs[0])
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_3.inputs[1])
                _nodegroup_node_3.location = Vector((500, -550))          

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMapRange") 
                _nodegroup_node_2.clamp = False
                _nodegroup_node_2.inputs[1].default_value = 0.0
                _nodegroup_node_2.inputs[2].default_value = 1.0
                _nodegroup_node_2.inputs[3].default_value = 1000.0
                _nodegroup_node_2.inputs[4].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_2.inputs[0])
                _nodegroup_node_2.location = Vector((700, -550))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'SUBTRACT'
                _nodegroup_node_3.inputs[0].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_3.inputs[1])
                _nodegroup_node_3.location = Vector((900, -400))

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_4.operation = 'ADD'
                _nodegroup_node_4.inputs[1].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_4.inputs[0])
                _nodegroup_node_4.location = Vector((900, -600))

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMapRange") 
                _nodegroup_node_2.clamp = False
                _nodegroup_node_2.inputs[1].default_value = 0.0
                _nodegroup_node_2.inputs[2].default_value = 1.0
                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_node_2.inputs[0])
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_2.inputs[3])
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_2.inputs[4])
                _nodegroup_node_2.location = Vector((1200, 125))

                _nodegroup_node_1 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_1.operation = 'ADD'
                _nodegroup_node_1.use_clamp = True
                _nodegroup_node_1.inputs[1].default_value = 0
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_1.inputs[0])
                _nodegroup_node_1.location = Vector((1400, 125))

                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_output.inputs[0])

            _nodegroup = for_nodes.new('ShaderNodeGroup') 
            _nodegroup.node_tree = bpy.data.node_groups[BLENDIT_NODEGROUP_CUSTOM]
            _nodegroup.name = BLENDIT_NODEGROUP_CUSTOM 
            return _nodegroup


        def create_nodegroup_noise(for_nodes): 

            if not BLENDIT_NODEGROUP_NOISE in bpy.data.node_groups: 
                _nodegroup = bpy.data.node_groups.new(name=BLENDIT_NODEGROUP_NOISE, type="ShaderNodeTree")

                _nodegroup_input = _nodegroup.nodes.new("NodeGroupInput") 
                _input =_nodegroup.inputs.new('NodeSocketFloatFactor', "Weight")
                _input.default_value = 0.5
                _nodegroup.inputs.new('NodeSocketFloat', "Seed")
                _input = _nodegroup.inputs.new('NodeSocketFloatFactor', "Amplitude")
                _input.min_value = -100.0
                _input.max_value = 100.0
                _input.default_value = 1.0
                _input = _nodegroup.inputs.new('NodeSocketFloatFactor', "Threshold")
                _input.min_value = -10.0
                _input.max_value = 10.0
                _input = _nodegroup.inputs.new('NodeSocketFloatFactor', "Frequency")
                _input.min_value = 0
                _input.max_value = 100.0
                _input.default_value = 1.0
                _input = _nodegroup.inputs.new('NodeSocketFloatFactor', "Detail")
                _input.min_value = 0
                _input.max_value = 16.0
                _input.default_value = 3.0
                _input = _nodegroup.inputs.new('NodeSocketFloatFactor', "Roughness")
                _input.min_value = 0
                _input.max_value = 1.0
                _input.default_value = 0.5
                _input = _nodegroup.inputs.new('NodeSocketFloatFactor', "Distortion")
                _input.min_value = -50.0
                _input.max_value = 50.0
                _input.default_value = 0
                _input = _nodegroup.inputs.new('NodeSocketFloatFactor', "Radius")
                _input.min_value = 0
                _input.max_value = 1.0
                _input.default_value = 1.0
                _nodegroup_input.location = Vector((-2000, 50))

                _nodegroup_output = _nodegroup.nodes.new("NodeGroupOutput") 
                _output = _nodegroup.outputs.new('NodeSocketFloatFactor', "Fac")
                _nodegroup_output.location = Vector((1700, 50))

                _nodegroup_node_1 = _nodegroup.nodes.new("ShaderNodeNewGeometry") 
                _nodegroup_node_1.location = Vector((-2000, -225))

                _nodegroup_node_5 = _nodegroup.nodes.new("ShaderNodeTexNoise") 
                _nodegroup_node_5.noise_dimensions = '4D'
                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_node_5.inputs[0])
                _nodegroup.links.new(_nodegroup_input.outputs[1], _nodegroup_node_5.inputs[1])
                _nodegroup.links.new(_nodegroup_input.outputs[4], _nodegroup_node_5.inputs[2])
                _nodegroup.links.new(_nodegroup_input.outputs[5], _nodegroup_node_5.inputs[3])
                _nodegroup.links.new(_nodegroup_input.outputs[6], _nodegroup_node_5.inputs[4])
                _nodegroup.links.new(_nodegroup_input.outputs[7], _nodegroup_node_5.inputs[5])
                _nodegroup_node_5.location = Vector((-1600, -100))

                _nodegroup_node_1 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_1.operation = 'DIVIDE'
                _nodegroup_node_1.inputs[1].default_value = 2.0
                _nodegroup.links.new(_nodegroup_input.outputs[2], _nodegroup_node_1.inputs[0])
                _nodegroup_node_1.location = Vector((-1400, 85))

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_2.operation = 'SUBTRACT'
                _nodegroup_node_2.inputs[0].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_node_2.inputs[1])
                _nodegroup_node_2.location = Vector((-1200, 150))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'ADD'
                _nodegroup_node_3.inputs[1].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_node_3.inputs[0])
                _nodegroup_node_3.location = Vector((-1200, -25))

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMapRange") 
                _nodegroup_node_4.clamp = False
                _nodegroup_node_4.inputs[1].default_value = 0.0
                _nodegroup_node_4.inputs[2].default_value = 1.0
                _nodegroup.links.new(_nodegroup_node_5.outputs[0], _nodegroup_node_4.inputs[0])
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_4.inputs[3])
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_4.inputs[4])
                _nodegroup_node_4.location = Vector((-1000, 110))

                _nodegroup_node_1 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_1.operation = 'ADD'
                _nodegroup_node_1.use_clamp = True
                _nodegroup_node_1.inputs[1].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_1.inputs[0])
                _nodegroup.links.new(_nodegroup_input.outputs[3], _nodegroup_node_1.inputs[1])
                _nodegroup_node_1.location = Vector((-800, 0))

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_2.operation = 'ADD'
                _nodegroup_node_2.use_clamp = True
                _nodegroup_node_2.inputs[1].default_value = 0
                _nodegroup.links.new(_nodegroup_input.outputs[0], _nodegroup_node_2.inputs[0])
                _nodegroup_node_2.location = Vector((-1000, 300))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'SUBTRACT'
                _nodegroup_node_3.inputs[0].default_value = 1.0
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_3.inputs[1])
                _nodegroup_node_3.location = Vector((-500, 450))                

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_4.operation = 'SUBTRACT'
                _nodegroup_node_4.inputs[0].default_value = 1.0
                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_node_4.inputs[1])
                _nodegroup_node_4.location = Vector((-500, 250))                

                _nodegroup_node_5 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_5.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_5.inputs[0])
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_5.inputs[1])
                _nodegroup_node_5.location = Vector((-300, 400))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_node_5.outputs[0], _nodegroup_node_3.inputs[0])
                _nodegroup_node_3.inputs[1].default_value = 2.0
                _nodegroup_node_3.location = Vector((-100, 400))

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_4.operation = 'SUBTRACT'
                _nodegroup_node_4.inputs[0].default_value = 1.0
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_4.inputs[1])
                _nodegroup_node_4.location = Vector((100, 400))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_3.inputs[0])
                _nodegroup_node_3.location = Vector((300, 300))

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_4.operation = 'GREATER_THAN'
                _nodegroup_node_4.use_clamp = True
                _nodegroup_node_4.inputs[1].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_4.inputs[0])
                _nodegroup_node_4.location = Vector((-100, 200))

                _nodegroup_node_5 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_5.operation = 'COMPARE'
                _nodegroup_node_5.use_clamp = True
                _nodegroup_node_5.inputs[1].default_value = 0.5
                _nodegroup_node_5.inputs[2].default_value = 0
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_5.inputs[0])
                _nodegroup_node_5.location = Vector((-100, 0))

                _nodegroup_node_6 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_6.operation = 'ADD'
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_6.inputs[0])
                _nodegroup.links.new(_nodegroup_node_5.outputs[0], _nodegroup_node_6.inputs[1])
                _nodegroup.links.new(_nodegroup_node_6.outputs[0], _nodegroup_node_3.inputs[1])
                _nodegroup_node_6.location = Vector((100, 150))

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_4.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_4.inputs[0])
                _nodegroup_node_4.inputs[1].default_value = 2.0
                _nodegroup_node_4.location = Vector((-100, -200))
                
                _nodegroup_node_5 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_5.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_5.inputs[0])
                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_node_5.inputs[1])
                _nodegroup_node_5.location = Vector((100, -250))

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_4.operation = 'LESS_THAN'
                _nodegroup_node_4.use_clamp = True
                _nodegroup_node_4.inputs[1].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_4.inputs[0])
                _nodegroup_node_4.location = Vector((100, -450))

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_2.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_node_5.outputs[0], _nodegroup_node_2.inputs[0])
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_2.inputs[1])
                _nodegroup_node_2.location = Vector((300, -250))

                _nodegroup_node_1 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_1.operation = 'ADD'
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_1.inputs[0])
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_1.inputs[1])
                _nodegroup_node_1.location = Vector((500, 50))

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMapRange") 
                _nodegroup_node_2.clamp = False
                _nodegroup_node_2.inputs[1].default_value = 0.0
                _nodegroup_node_2.inputs[2].default_value = 1.0
                _nodegroup_node_2.inputs[3].default_value = 0.01
                _nodegroup_node_2.inputs[4].default_value = 0.0001
                _nodegroup.links.new(_nodegroup_input.outputs[8], _nodegroup_node_2.inputs[0])
                _nodegroup_node_2.location = Vector((300, -700))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'POWER'
                _nodegroup.links.new(_nodegroup_input.outputs[8], _nodegroup_node_3.inputs[0])
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_3.inputs[1])
                _nodegroup_node_3.location = Vector((500, -550))          

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMapRange") 
                _nodegroup_node_2.clamp = False
                _nodegroup_node_2.inputs[1].default_value = 0.0
                _nodegroup_node_2.inputs[2].default_value = 1.0
                _nodegroup_node_2.inputs[3].default_value = 1000.0
                _nodegroup_node_2.inputs[4].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_2.inputs[0])
                _nodegroup_node_2.location = Vector((700, -550))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'SUBTRACT'
                _nodegroup_node_3.inputs[0].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_3.inputs[1])
                _nodegroup_node_3.location = Vector((900, -400))

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_4.operation = 'ADD'
                _nodegroup_node_4.inputs[1].default_value = 0.5
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_4.inputs[0])
                _nodegroup_node_4.location = Vector((900, -600))

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMapRange") 
                _nodegroup_node_2.clamp = False
                _nodegroup_node_2.inputs[1].default_value = 0.0
                _nodegroup_node_2.inputs[2].default_value = 1.0
                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_node_2.inputs[0])
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_2.inputs[3])
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_2.inputs[4])
                _nodegroup_node_2.location = Vector((1200, 125))

                _nodegroup_node_1 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_1.operation = 'ADD'
                _nodegroup_node_1.use_clamp = True
                _nodegroup_node_1.inputs[1].default_value = 0
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_1.inputs[0])
                _nodegroup_node_1.location = Vector((1400, 125))

                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_output.inputs[0])

            _nodegroup = for_nodes.new('ShaderNodeGroup') 
            _nodegroup.node_tree = bpy.data.node_groups[BLENDIT_NODEGROUP_NOISE]
            _nodegroup.name = BLENDIT_NODEGROUP_NOISE 
            return _nodegroup


        def create_nodegroup_height(for_nodes): 

            if not BLENDIT_NODEGROUP_HEIGHT in bpy.data.node_groups: 
                _nodegroup = bpy.data.node_groups.new(name=BLENDIT_NODEGROUP_HEIGHT, type="ShaderNodeTree")

                _nodegroup_input = _nodegroup.nodes.new("NodeGroupInput") 
                _input =_nodegroup.inputs.new('NodeSocketFloatFactor', "Weight")
                _input.default_value = 0.5
                _nodegroup.inputs.new('NodeSocketFloat', "Height 0")
                _input = _nodegroup.inputs.new('NodeSocketFloatFactor', "Scale")
                _input.min_value = 0
                _input.max_value = 100.0
                _input.default_value = 1.0
                _input = _nodegroup.inputs.new('NodeSocketFloatFactor', "Threshold")
                _input.min_value = -10.0
                _input.max_value = 10.0
                _nodegroup.inputs.new('NodeSocketFloat', "Height 1")
                _input = _nodegroup.inputs.new('NodeSocketFloatFactor', "Scale")
                _input.min_value = 0
                _input.max_value = 100.0
                _input.default_value = 1.0
                _input = _nodegroup.inputs.new('NodeSocketFloatFactor', "Threshold")
                _input.min_value = -10.0
                _input.max_value = 10.0
                _input = _nodegroup.inputs.new('NodeSocketFloatFactor', "Radius")
                _input.min_value = 0
                _input.max_value = 1.0
                _input.default_value = 1.0
                _nodegroup_input.location = Vector((-1700, -100))

                _nodegroup_output = _nodegroup.nodes.new("NodeGroupOutput") 
                _output = _nodegroup.outputs.new('NodeSocketFloatFactor', "Fac")
                _nodegroup_output.location = Vector((1500, 50))

                _nodegroup_node_1 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_1.operation = 'ADD'
                _nodegroup_node_1.use_clamp = True
                _nodegroup_node_1.inputs[1].default_value = 0
                _nodegroup.links.new(_nodegroup_input.outputs[0], _nodegroup_node_1.inputs[0])
                _nodegroup_node_1.location = Vector((-400, 300))

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_2.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_input.outputs[1], _nodegroup_node_2.inputs[0])
                _nodegroup.links.new(_nodegroup_input.outputs[2], _nodegroup_node_2.inputs[1])
                _nodegroup_node_2.location = Vector((-1250, 125))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'ADD'
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_3.inputs[0])
                _nodegroup.links.new(_nodegroup_input.outputs[3], _nodegroup_node_3.inputs[1])
                _nodegroup_node_3.location = Vector((-1050, 50))

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_4.operation = 'MAXIMUM'
                _nodegroup_node_4.inputs[1].default_value = 0
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_4.inputs[0])
                _nodegroup_node_4.location = Vector((-850, 50))

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_2.operation = 'ADD'
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_2.inputs[0])
                _nodegroup_node_2.location = Vector((-400, 50))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_input.outputs[4], _nodegroup_node_3.inputs[0])
                _nodegroup.links.new(_nodegroup_input.outputs[5], _nodegroup_node_3.inputs[1])
                _nodegroup_node_3.location = Vector((-1250, -125))

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_4.operation = 'ADD'
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_4.inputs[0])
                _nodegroup.links.new(_nodegroup_input.outputs[6], _nodegroup_node_4.inputs[1])
                _nodegroup_node_4.location = Vector((-1050, -200))

                _nodegroup_node_5 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_5.operation = 'MAXIMUM'
                _nodegroup_node_5.inputs[1].default_value = 0
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_5.inputs[0])
                _nodegroup_node_5.location = Vector((-850, -200))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'ADD'
                _nodegroup.links.new(_nodegroup_node_5.outputs[0], _nodegroup_node_3.inputs[0])
                _nodegroup_node_3.location = Vector((-400, -200))

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMapRange") 
                _nodegroup_node_4.clamp = True
                _nodegroup_node_4.inputs[1].default_value = 0.0
                _nodegroup_node_4.inputs[2].default_value = 1.0
                _nodegroup_node_4.inputs[3].default_value = 0.00001
                _nodegroup_node_4.inputs[4].default_value = 1.0
                _nodegroup.links.new(_nodegroup_input.outputs[7], _nodegroup_node_4.inputs[0])
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_2.inputs[1])
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_3.inputs[1])
                _nodegroup_node_4.location = Vector((-700, -400))

                _nodegroup_node_6 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_6.operation = 'SUBTRACT'
                _nodegroup_node_6.inputs[0].default_value = 1.0
                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_node_6.inputs[1])
                _nodegroup_node_6.location = Vector((-200, 200))

                _nodegroup_node_5 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_5.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_node_6.outputs[0], _nodegroup_node_5.inputs[0])
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_5.inputs[1])
                _nodegroup_node_5.location = Vector((0, 150))

                _nodegroup_node_6 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_6.operation = 'MULTIPLY'
                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_node_6.inputs[0])
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_6.inputs[1])
                _nodegroup_node_6.location = Vector((-200, -100))

                _nodegroup_node_1 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_1.operation = 'MAXIMUM'
                _nodegroup.links.new(_nodegroup_node_5.outputs[0], _nodegroup_node_1.inputs[0])
                _nodegroup.links.new(_nodegroup_node_6.outputs[0], _nodegroup_node_1.inputs[1])
                _nodegroup_node_1.location = Vector((200, 100))

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_2.operation = 'SUBTRACT'
                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_node_2.inputs[0])
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_2.inputs[1])
                _nodegroup_node_2.location = Vector((400, 50))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'SUBTRACT'
                _nodegroup.links.new(_nodegroup_node_5.outputs[0], _nodegroup_node_3.inputs[0])
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_3.inputs[1])
                _nodegroup_node_3.location = Vector((600, 235))

                _nodegroup_node_4 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_4.operation = 'SUBTRACT'
                _nodegroup.links.new(_nodegroup_node_6.outputs[0], _nodegroup_node_4.inputs[0])
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_4.inputs[1])
                _nodegroup_node_4.location = Vector((600, -25))

                _nodegroup_node_1 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_1.operation = 'MAXIMUM'
                _nodegroup_node_1.inputs[0].default_value = 0.0
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_1.inputs[1])
                _nodegroup_node_1.location = Vector((800, 235))

                _nodegroup_node_2 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_2.operation = 'MAXIMUM'
                _nodegroup_node_2.inputs[0].default_value = 0.0
                _nodegroup.links.new(_nodegroup_node_4.outputs[0], _nodegroup_node_2.inputs[1])
                _nodegroup_node_2.location = Vector((800, -25))

                _nodegroup_node_3 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_3.operation = 'ADD'
                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_node_3.inputs[0])
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_3.inputs[1])
                _nodegroup_node_3.location = Vector((1000, -100))

                _nodegroup_node_1 = _nodegroup.nodes.new("ShaderNodeMath") 
                _nodegroup_node_1.operation = 'DIVIDE'
                _nodegroup.links.new(_nodegroup_node_2.outputs[0], _nodegroup_node_1.inputs[0])
                _nodegroup.links.new(_nodegroup_node_3.outputs[0], _nodegroup_node_1.inputs[1])
                _nodegroup_node_1.location = Vector((1200, 50))

                _nodegroup.links.new(_nodegroup_node_1.outputs[0], _nodegroup_output.inputs[0])

            _nodegroup = for_nodes.new('ShaderNodeGroup') 
            _nodegroup.node_tree = bpy.data.node_groups[BLENDIT_NODEGROUP_HEIGHT]
            _nodegroup.name = BLENDIT_NODEGROUP_HEIGHT 
            return _nodegroup

        def relink_blend_style(object): 
            _node_tree = object.material_slots[0].material.node_tree 
            _reroutes = []

            if object.blendit_locals.blend_style == 2 or object.blendit_locals.blend_style == 3:

                if object.blendit_locals.blend_style == 2: 
                    _style_node = _node_tree.nodes[BLENDIT_ID_NAME + '_Heights_Blend'] 
                    if _style_node.inputs[1].is_linked:
                        _node_tree.links.remove(_style_node.inputs[1].links[0])
                    if _style_node.inputs[4].is_linked:
                        _node_tree.links.remove(_style_node.inputs[4].links[0])
                    _loops = 2

                if object.blendit_locals.blend_style == 3: 
                    _style_node = _node_tree.nodes[BLENDIT_ID_NAME + '_Custom_Blend'] 
                    if _style_node.inputs[1].is_linked:
                        _node_tree.links.remove(_style_node.inputs[1].links[0])
                    _loops = 1

                for _node in _node_tree.nodes: 
                    if _node.bl_idname == "NodeReroute" and _node.label == _node_tree.blendit_material.style_reroute:
                        _reroutes.append(_node)
                        _loops -= 1
                        if _loops == 0:
                            break
                        
                if _reroutes: 
                    for indx, _reroute in enumerate(_reroutes):
                        _node_tree.links.new(_reroute.outputs[0], _style_node.inputs[1+(indx*3)]) 
            return

        def create_material_blend(_obj_source, _obj_target):
            _origin_mat = _obj_target.material_slots[0].material 
            _source_mat = _obj_source.material_slots[0].material
            _created_new_mat = False
            _offset = 100
            _node_out1 = None
            _node_out2 = None

            if _obj_target.blendit_locals.advanced_mod:
                _mat_id_name = BLENDIT_ID_NAME + "_A" 
            else:
                _mat_id_name = BLENDIT_ID_NAME + "_S" 

            if _obj_target.blendit_locals.blend_style == 0: 
                _mat_id_name += "_"
            elif _obj_target.blendit_locals.blend_style == 1: 
                _mat_id_name += "N_"
            elif _obj_target.blendit_locals.blend_style == 2: 
                _mat_id_name += "H_"
            elif _obj_target.blendit_locals.blend_style == 3: 
                _mat_id_name += "C_"

            if _origin_mat and _origin_mat.use_nodes and _source_mat and _source_mat.use_nodes: 
                if (_origin_mat.name[:len(_mat_id_name)] != _mat_id_name): 
                    bpy.ops.outliner.orphans_purge() 
                    
                    if  ((not (_mat_id_name + _origin_mat.name) in bpy.data.materials)  
                        or (bpy.data.materials[_mat_id_name + _origin_mat.name].users == 0)):

                        _origin_mat.use_fake_user = True 
                        _new_mat = _origin_mat.copy() 
                        _new_mat.name = _mat_id_name + _origin_mat.name
                        _new_mat.use_nodes = True
                        
                        _origin_offset_min, _origin_offset_max = get_nodes_borders(_origin_mat.node_tree.nodes)
                        _source_offset_min, _source_offset_max = get_nodes_borders(_source_mat.node_tree.nodes)
                        _nodes_offset = Vector((_origin_offset_max[0]-_source_offset_max[0], _origin_offset_min[1]-_source_offset_max[1]-500))
                        
                        for _node in _new_mat.node_tree.nodes: 
                            _node.name = "OriginNode_" + _node.name

                        for _node in _new_mat.node_tree.nodes:
                            if _node.bl_idname == "ShaderNodeOutputMaterial" and _node.is_active_output and _node.inputs[0].is_linked:
                                _node_out1 = _node
                                break
                        
                        for _node in _source_mat.node_tree.nodes: 
                            _node_clon = copy_node(_node, _new_mat.node_tree.nodes)
                            if _node_clon.parent:
                                _node_clon.location = get_node_location(_node_clon)
                            else:
                                _node_clon.location = _node_clon.location + _nodes_offset
                            if _node.bl_idname == "ShaderNodeTexCoord" and _node.object == None: 
                                _node_clon.object = _obj_source

                            if _node.bl_idname == "ShaderNodeUVMap": 
                                if active_uv:
                                    _node_clon.uv_map = active_uv
                                else: 
                                    _node_clon.uv_map = ""

                            if _node.bl_idname == "ShaderNodeNormalMap": 
                                if active_uv:
                                    _node_clon.uv_map = active_uv
                                else: 
                                    _node_clon.uv_map = ""

                            if not _node_out2 and _node.bl_idname == "ShaderNodeOutputMaterial" and _node.is_active_output and _node.inputs[0].is_linked:
                                _node_out2 = _node_clon

                        copy_nodes_links(_source_mat.node_tree.nodes, _new_mat.node_tree, active_uv)
                        
                        connect_unconnected_nodes(_source_mat.node_tree.nodes, _new_mat.node_tree, active_uv) 

                        _node_vec = _new_mat.node_tree.nodes.new("ShaderNodeVertexColor")
                        _node_vec.layer_name = BLENDIT_BLEND_ID_NAME 
                        _node_vec.location = Vector((_origin_offset_max[0] + _offset, _origin_offset_min[1]-300))

                        if _obj_target.blendit_locals.advanced_mod: 
                            
                            _node_sub = _new_mat.node_tree.nodes.new("ShaderNodeMath")
                            _node_sub.operation = "SUBTRACT"
                            _node_sub.inputs[0].default_value = 1.0
                            _offset += 200
                            _node_sub.location = Vector((_origin_offset_max[0] + _offset, _origin_offset_min[1]-300))
                            
                            _node_cor = _new_mat.node_tree.nodes.new("ShaderNodeValToRGB")
                            _node_cor.color_ramp.elements.new(0.5)
                            _node_cor.color_ramp.elements[0].color = (1.0, 1.0, 1.0, 1.0)
                            _node_cor.color_ramp.elements[1].color = (0.5, 0.5, 0.5, 1.0)
                            _node_cor.color_ramp.elements[2].color = (0.0, 0.0, 0.0, 1.0)
                            _node_cor.name = BLENDIT_ID_NAME + "ColorRamp"
                            _offset += 200
                            _node_cor.location = Vector((_origin_offset_max[0] + _offset, _origin_offset_min[1]-300))

                            _offset += 100
                            
                        if _obj_target.blendit_locals.blend_style == 1: 
                            _node_style = create_nodegroup_noise(_new_mat.node_tree.nodes)
                            _offset += 200
                            _node_style.location = Vector((_origin_offset_max[0] + _offset, _origin_offset_min[1]-300))
                            
                            _node_style.inputs[1].default_value = _obj_source.blendit_locals.style_seed
                            _node_style.inputs[2].default_value = _obj_source.blendit_locals.style_amplitude
                            _node_style.inputs[3].default_value = _obj_source.blendit_locals.style_brightness
                            _node_style.inputs[4].default_value = _obj_source.blendit_locals.style_frequency
                            _node_style.inputs[5].default_value = _obj_source.blendit_locals.style_detail
                            _node_style.inputs[6].default_value = _obj_source.blendit_locals.style_roughness
                            _node_style.inputs[7].default_value = _obj_source.blendit_locals.style_distortion
                            _node_style.inputs[8].default_value = _obj_source.blendit_locals.style_smooth

                        elif _obj_target.blendit_locals.blend_style == 2: 
                            _node_style = create_nodegroup_height(_new_mat.node_tree.nodes)
                            _offset += 200
                            _node_style.location = Vector((_origin_offset_max[0] + _offset, _origin_offset_min[1]-300))
                            
                            _node_style.inputs[2].default_value = _obj_source.blendit_locals.style_scale
                            _node_style.inputs[3].default_value = _obj_source.blendit_locals.style_brightness
                            _node_style.inputs[5].default_value = _obj_source.blendit_locals.style_scale2
                            _node_style.inputs[6].default_value = _obj_source.blendit_locals.style_brightness2
                            _node_style.inputs[7].default_value = _obj_source.blendit_locals.style_smooth
                            
                        elif _obj_target.blendit_locals.blend_style == 3: 
                            _node_style = create_nodegroup_custom(_new_mat.node_tree.nodes)
                            _offset += 200
                            _node_style.location = Vector((_origin_offset_max[0] + _offset, _origin_offset_min[1]-300))
                            
                            _node_style.inputs[2].default_value = _obj_source.blendit_locals.style_amplitude
                            _node_style.inputs[3].default_value = _obj_source.blendit_locals.style_brightness
                            _node_style.inputs[4].default_value = _obj_source.blendit_locals.style_smooth
                        
                        _node_mix = create_nodegroup_mixshader(_new_mat.node_tree.nodes)
                        
                        _offset += 250
                        _node_mix.location = Vector((_origin_offset_max[0] + _offset, _origin_offset_min[1]-500))
                        
                        _node_out = _new_mat.node_tree.nodes.new("ShaderNodeOutputMaterial")
                        _offset += 200
                        _node_out.location = Vector((_origin_offset_max[0] + _offset, _origin_offset_min[1]-500))
                        _node_out.is_active_output = True
                        if _node_out1:
                            _node_out1.is_active_output = False
                        if _node_out2:
                            _node_out2.is_active_output = False

                        if _node_out1:
                            _link = _node_out1.inputs[0].links[0] 
                            _new_mat.node_tree.links.new(_new_mat.node_tree.nodes[_link.from_node.name].outputs[_link.from_socket.name], _node_mix.inputs[1])

                        if _node_out2:
                            _link = _node_out2.inputs[0].links[0] 
                            _new_mat.node_tree.links.new(_new_mat.node_tree.nodes[_link.from_node.name].outputs[_link.from_socket.name], _node_mix.inputs[2])

                        if _obj_target.blendit_locals.advanced_mod:
                            
                            _new_mat.node_tree.links.new(_node_vec.outputs[0], _node_sub.inputs[1])
                            
                            _new_mat.node_tree.links.new(_node_sub.outputs[0], _node_cor.inputs[0])
                            if _obj_target.blendit_locals.blend_style == 0:
                                
                                _new_mat.node_tree.links.new(_node_cor.outputs[0], _node_mix.inputs[0])
                            else: 
                                _new_mat.node_tree.links.new(_node_cor.outputs[0], _node_style.inputs[0])
                                _new_mat.node_tree.links.new(_node_style.outputs[0], _node_mix.inputs[0])
                        else:
                            if _obj_target.blendit_locals.blend_style == 0:
                                
                                _new_mat.node_tree.links.new(_node_vec.outputs[0], _node_mix.inputs[0])
                            else: 
                                _new_mat.node_tree.links.new(_node_vec.outputs[0], _node_style.inputs[0])
                                _new_mat.node_tree.links.new(_node_style.outputs[0], _node_mix.inputs[0])

                        _new_mat.node_tree.links.new(_node_mix.outputs[0], _node_out.inputs[0])
                        
                        _obj_target.material_slots[0].material = _new_mat 
                        _new_mat.node_tree.blendit_material.style_reroute = _obj_target.blendit_locals.style_reroute 
                        relink_blend_style(_obj_target) 
                        
                        _created_new_mat = True
                        
                        print("Saving mat: ", _new_mat.name)
                    else: 
                        print("Mat is in list: ", _mat_id_name + _origin_mat.name)
                        _new_mat = bpy.data.materials[_mat_id_name + _origin_mat.name]
                        _obj_target.material_slots[0].material = _new_mat 
                    
                else:
                    print("Mat existed: ", _mat_id_name + _origin_mat.name)
            return _created_new_mat
        
        print("Blendit: Triggered Create Blends")
        scene = context.scene
        blendit_globals = scene.blendit_globals
        active_obj = context.object
        created_new = False
        active_render_uv = None 
        active_uv = None 
        indx = 0
        warning = 0
        has_mat_source = False
        has_mat = False
        source_obj = None
        mat_missing = False
        
        if self.action == "relink": 
            relink_blend_style(active_obj)

        elif self.action == "switch": 
            obj = active_obj
            
            if obj.blendit_locals.blend_style == 2: 
                obj.blendit_locals.style_reroute = "height"
            elif obj.blendit_locals.blend_style == 3: 
                obj.blendit_locals.style_reroute = "custom"

            if obj.blendit_locals.blended and not obj.blendit_locals.blend_source:
                source_obj = obj.blendit_locals.source_obj 
                blendit_locals = obj.blendit_locals

                create_drivers(source_obj, obj)

                try: 
                    has_mat_source = bool(source_obj.material_slots[0].material) 
                    has_mat = bool(obj.material_slots[0].material) 
                except:
                    has_mat_source = False
                    has_mat = False

                if has_mat and has_mat_source:
                    if obj.material_slots[0].material.name[:len(BLENDIT_ID_NAME)] == BLENDIT_ID_NAME: 
                        if source_obj.data.uv_layers and blendit_globals.transfer_uv:
                            active_uv = BLENDIT_ID_NAME
                        elif obj.data.uv_layers and not blendit_globals.transfer_uv: 
                            active_uv = BLENDIT_ID_NAME

                        _mat_name = obj.material_slots[0].material.name[len(BLENDIT_ID_NAME)+3:] 
                        if not _mat_name in bpy.data.materials: 
                            _mat_name = obj.material_slots[0].material.name[len(BLENDIT_ID_NAME)+4:] 
                            if not _mat_name in bpy.data.materials: 
                                _mat_name = obj.material_slots[0].material.name[len(BLENDIT_ID_NAME):] 
                                if not _mat_name in bpy.data.materials: 
                                    _mat_name = None 
                            
                        if _mat_name: 
                            obj.material_slots[0].material.use_fake_user = False
                            obj.material_slots[0].material = bpy.data.materials[_mat_name] 
                            create_material_blend(source_obj, obj) 
                        else:
                            bpy.ops.blendit.dialog_warning('INVOKE_DEFAULT', warning = 0b1000)

        else: 
            if 1 < len(context.selected_objects) and not active_obj.blendit_locals.blended:
                try: 
                    has_mat = bool(active_obj.material_slots[0].material)
                except:
                    has_mat = False

                if has_mat or not blendit_globals.new_mat: 
                    if active_obj.data.uv_layers and blendit_globals.transfer_uv:
                        for indx, uv_layer in enumerate(bpy.data.meshes[bpy.data.objects[active_obj.name].data.name].uv_layers): 
                            if uv_layer.active_render:
                                active_render_uv = uv_layer.name
                                break
                        active_obj.data.uv_layers.active_index = indx 
                        active_uv = active_obj.data.uv_layers[active_obj.data.uv_layers.active_index].name
                        active_uv = duplicate_mapping(active_obj, active_uv, BLENDIT_ID_NAME).name 
                    else:
                        active_uv = None

                    create_vertex_color(active_obj, BLENDIT_BLEND_ID_NAME, 1)
                    active_obj.blendit_locals.blend_source = True
                    
                    for obj in context.selected_objects:
                        if (obj != active_obj and
                            isinstance(obj, bpy.types.Object) and
                            isinstance(obj.data, bpy.types.Mesh) and
                            not obj.blendit_locals.blend_source and
                            not obj.blendit_locals.blended
                            ):
                            if blendit_globals.new_mat:
                                try: 
                                    has_mat = bool(obj.material_slots[0].material)
                                    if not has_mat: 
                                        mat_missing = True
                                except: 
                                    has_mat = False
                                    mat_missing = True

                            if has_mat or not blendit_globals.new_mat:
                                blendit_locals = obj.blendit_locals

                                blendit_locals.advanced_mod = active_obj.blendit_locals.advanced_mod 
                                blendit_locals.blend_style = active_obj.blendit_locals.blend_style
                                blendit_locals.style_reroute = active_obj.blendit_locals.style_reroute
                                
                                create_vertex_group(obj, BLENDIT_ID_NAME)
                                create_vertex_color(obj, BLENDIT_BLEND_ID_NAME, 0)

                                if active_obj.data.uv_layers and blendit_globals.transfer_uv: 
                                    create_mapping(obj, BLENDIT_ID_NAME)
                                elif obj.data.uv_layers and not blendit_globals.transfer_uv: 
                                    for indx, uv_layer in enumerate(bpy.data.meshes[bpy.data.objects[obj.name].data.name].uv_layers): 
                                        if uv_layer.active_render:
                                            active_render_uv = uv_layer.name
                                            break
                                    obj.data.uv_layers.active_index = indx 
                                    active_uv = obj.data.uv_layers[obj.data.uv_layers.active_index].name
                                    active_uv = duplicate_mapping(obj, active_uv, BLENDIT_ID_NAME).name

                                create_modifiers(obj, BLENDIT_ID_NAME)
                                create_drivers(active_obj, obj)
                                if blendit_globals.new_mat:
                                    if create_material_blend(active_obj, obj):
                                        created_new = True
                                
                                blendit_locals.source_obj = active_obj
                                blendit_locals.blended = True

                    if created_new: 
                        warning = check_material_nodes(active_obj.material_slots[0].material.node_tree.nodes) 
                    if mat_missing: 
                        warning += 0b001
                    if warning:
                        bpy.ops.blendit.dialog_warning('INVOKE_DEFAULT', warning = warning)
                else: 
                    bpy.ops.blendit.dialog_warning('INVOKE_DEFAULT', warning = 0b001)
        print("Finished")
        return {'FINISHED'}


class Blendit_OT_Apply(bpy.types.Operator):
    bl_label = "Apply Dynamic Blends"
    bl_idname = "blendit.apply"
    bl_description = "Apply dynamic blend modifiers on selected objects"

    @classmethod
    def poll(cls, context):
        scene = context.scene
        p = (context.mode == 'OBJECT' and
            (context.object.blendit_locals.blended
            
            or len(context.selected_objects)>1)
            )
        return p

    def execute(self, context):
        print("Triggered Apply Blends")
        active_obj = context.object
        indx = len(context.selected_objects)
        while indx > 0:
            indx-=1
            obj = context.selected_objects[indx]
            blendit_locals = obj.blendit_locals
            temporarily_auto_smooth_changed = False
            
            if blendit_locals.blended:
                blendit_locals.applied = True
                bpy.context.view_layer.objects.active = obj 

                _mod_name = BLENDIT_ID_NAME + "_V_W_Proximity"
                if _mod_name in obj.modifiers:
                    
                    _min_dist = obj.modifiers[BLENDIT_ID_NAME+"_V_W_Proximity"].min_dist
                    _max_dist = obj.modifiers[BLENDIT_ID_NAME+"_V_W_Proximity"].max_dist
                    if obj.animation_data and obj.animation_data.drivers: 
                        for _driver in obj.animation_data.drivers:
                            if _driver.data_path == 'modifiers["'+BLENDIT_ID_NAME+'_V_W_Proximity"].min_dist':
                                obj.modifiers[BLENDIT_ID_NAME+"_V_W_Proximity"].driver_remove("min_dist")
                            if _driver.data_path == 'modifiers["'+BLENDIT_ID_NAME+'_V_W_Proximity"].max_dist':
                                obj.modifiers[BLENDIT_ID_NAME+"_V_W_Proximity"].driver_remove("max_dist")

                    obj.modifiers[BLENDIT_ID_NAME+"_V_W_Proximity"].min_dist = _min_dist
                    obj.modifiers[BLENDIT_ID_NAME+"_V_W_Proximity"].max_dist = _max_dist
                    bpy.ops.object.modifier_apply(modifier=_mod_name)

                _mod_name = BLENDIT_ID_NAME + "_D_Transfer_Vcol"
                if _mod_name in obj.modifiers:
                    bpy.ops.object.modifier_apply(modifier=_mod_name)
                
                _mod_name = BLENDIT_ID_NAME + "_D_Transfer_Norm"
                if _mod_name in obj.modifiers:
                    if not obj.data.use_auto_smooth: 
                        obj.data.use_auto_smooth = True
                        obj.data.auto_smooth_angle = 3.14159
                        temporarily_auto_smooth_changed = True

                    _mix_factor = obj.modifiers[BLENDIT_ID_NAME+"_D_Transfer_Norm"].mix_factor
                    if obj.animation_data and obj.animation_data.drivers: 
                        for _driver in obj.animation_data.drivers:
                            if _driver.data_path == 'modifiers["'+BLENDIT_ID_NAME+'_D_Transfer_Norm"].mix_factor':
                                obj.modifiers[BLENDIT_ID_NAME+"_D_Transfer_Norm"].driver_remove("mix_factor")

                    obj.modifiers[BLENDIT_ID_NAME+"_D_Transfer_Norm"].mix_factor = _mix_factor
                    bpy.ops.object.modifier_apply(modifier=_mod_name)
                    blendit_locals.blended_normals = True
                    blendit_locals.blend_normals = False

                _mod_name = BLENDIT_ID_NAME + "_D_Transfer_UV"
                if _mod_name in obj.modifiers:
                    bpy.ops.object.modifier_apply(modifier=_mod_name)

                _mod_name = BLENDIT_ID_NAME + "_UV_Warp"
                if _mod_name in obj.modifiers:
                    
                    _uv_scale = obj.modifiers[BLENDIT_ID_NAME+"_UV_Warp"].scale
                    if obj.animation_data and obj.animation_data.drivers: 
                        for _driver in obj.animation_data.drivers:
                            if _driver.data_path == 'modifiers["'+BLENDIT_ID_NAME+'_UV_Warp"].scale':
                                obj.modifiers[BLENDIT_ID_NAME+"_UV_Warp"].driver_remove("scale")

                    obj.modifiers[BLENDIT_ID_NAME+"_UV_Warp"].scale = _uv_scale
                    bpy.ops.object.modifier_apply(modifier=_mod_name)
                    blendit_locals.reused_uv = False

                if temporarily_auto_smooth_changed: 
                    obj.data.use_auto_smooth = False 
                
                if BLENDIT_ID_NAME in obj.vertex_groups: 
                    obj.vertex_groups.remove(obj.vertex_groups[BLENDIT_ID_NAME])

        bpy.context.view_layer.objects.active = active_obj 
        return {'FINISHED'}

class Blendit_OT_Remove(bpy.types.Operator):
    bl_label = "Remove Blends"
    bl_idname = "blendit.remove"
    bl_description = "Remove blends from selected objects"

    @classmethod
    def poll(cls, context):
        scene = context.scene
        p = (context.mode == 'OBJECT' and
            ((context.object.blendit_locals.blend_source or context.object.blendit_locals.blended)
            
            or len(context.selected_objects)>1)
            )
        return p

    def execute(self, context):
        print("Triggered Remove Blends")
        active_obj = context.object
        indx = len(context.selected_objects)
        has_mat_source = False
        has_mat = False
        source_obj = None
        error = False

        while indx > 0:
            indx-=1
            obj = context.selected_objects[indx]
            blendit_locals = obj.blendit_locals
            if blendit_locals.blend_source:

                for _obj in bpy.data.objects: 
                    if _obj.blendit_locals.blended and _obj.blendit_locals.source_obj == obj:
                        try:
                            _obj.select_set(True)
                        except: 
                            error = True

                if not error:
                    
                    if BLENDIT_ID_NAME in obj.vertex_groups: 
                        obj.vertex_groups.remove(obj.vertex_groups[BLENDIT_ID_NAME])

                    if BLENDIT_BLEND_ID_NAME in obj.data.vertex_colors:
                        obj.data.vertex_colors.remove(obj.data.vertex_colors[BLENDIT_BLEND_ID_NAME])

                    if BLENDIT_ID_NAME in obj.data.uv_layers: 
                        bpy.context.view_layer.objects.active = obj 
                        obj.data.uv_layers[BLENDIT_ID_NAME].active = True
                        bpy.ops.mesh.uv_texture_remove()

                    blendit_locals.blend_source = False
                else:
                    bpy.ops.blendit.dialog_warning('INVOKE_DEFAULT', warning = 0b10000)

        indx = len(context.selected_objects)
        while indx > 0:
            indx-=1
            obj = context.selected_objects[indx]
            blendit_locals = obj.blendit_locals
            if blendit_locals.blended:
                blendit_locals.applied = False
                
                _mod_name = BLENDIT_ID_NAME + "_V_W_Proximity"
                if _mod_name in obj.modifiers:
                    
                    if obj.animation_data and obj.animation_data.drivers: 
                        for _driver in obj.animation_data.drivers:
                            if _driver.data_path == 'modifiers["'+BLENDIT_ID_NAME+'_V_W_Proximity"].min_dist':
                                obj.modifiers[BLENDIT_ID_NAME+"_V_W_Proximity"].driver_remove("min_dist")
                            if _driver.data_path == 'modifiers["'+BLENDIT_ID_NAME+'_V_W_Proximity"].max_dist':
                                obj.modifiers[BLENDIT_ID_NAME+"_V_W_Proximity"].driver_remove("max_dist")

                    obj.modifiers.remove(obj.modifiers[_mod_name])

                _mod_name = BLENDIT_ID_NAME + "_D_Transfer_Vcol"
                if _mod_name in obj.modifiers:
                    obj.modifiers.remove(obj.modifiers[_mod_name])
                
                _mod_name = BLENDIT_ID_NAME + "_D_Transfer_Norm"
                if _mod_name in obj.modifiers:
                    
                    if obj.animation_data and obj.animation_data.drivers: 
                        for _driver in obj.animation_data.drivers:
                            if _driver.data_path == 'modifiers["'+BLENDIT_ID_NAME+'_D_Transfer_Norm"].mix_factor':
                                obj.modifiers[BLENDIT_ID_NAME+"_D_Transfer_Norm"].driver_remove("mix_factor")

                    obj.modifiers.remove(obj.modifiers[_mod_name])

                _mod_name = BLENDIT_ID_NAME + "_UV_Warp"
                if _mod_name in obj.modifiers:
                    
                    if obj.animation_data and obj.animation_data.drivers: 
                        for _driver in obj.animation_data.drivers:
                            if _driver.data_path == 'modifiers["'+BLENDIT_ID_NAME+'_UV_Warp"].scale':
                                obj.modifiers[BLENDIT_ID_NAME+"_UV_Warp"].driver_remove("scale")

                    obj.modifiers.remove(obj.modifiers[_mod_name])
                    blendit_locals.reused_uv = False

                if obj.blendit_locals.auto_smooth_changed: 
                    obj.data.use_auto_smooth = False 
                    obj.blendit_locals.auto_smooth_changed = False

                _mod_name = BLENDIT_ID_NAME + "_D_Transfer_UV"
                if _mod_name in obj.modifiers:
                    obj.modifiers.remove(obj.modifiers[_mod_name])
                
                source_obj = obj.blendit_locals.source_obj 
                try: 
                    has_mat_source = bool(source_obj.material_slots[0].material) 
                    has_mat = bool(obj.material_slots[0].material) 
                except:
                    has_mat_source = False
                    has_mat = False

                if has_mat and has_mat_source:
                    if obj.material_slots[0].material.name[:len(BLENDIT_ID_NAME)] == BLENDIT_ID_NAME: 
                        _mat_name = obj.material_slots[0].material.name[len(BLENDIT_ID_NAME)+3:] 
                        if not _mat_name in bpy.data.materials: 
                            _mat_name = obj.material_slots[0].material.name[len(BLENDIT_ID_NAME)+4:] 
                            if not _mat_name in bpy.data.materials: 
                                _mat_name = obj.material_slots[0].material.name[len(BLENDIT_ID_NAME):] 
                                if not _mat_name in bpy.data.materials: 
                                    _mat_name = None 
                            
                        if _mat_name: 
                            obj.material_slots[0].material.use_fake_user = False
                            obj.material_slots[0].material = bpy.data.materials[_mat_name] 
                        else:
                            bpy.ops.blendit.dialog_warning('INVOKE_DEFAULT', warning = 0b1000)

                if BLENDIT_ID_NAME in obj.vertex_groups: 
                    obj.vertex_groups.remove(obj.vertex_groups[BLENDIT_ID_NAME])

                if BLENDIT_BLEND_ID_NAME in obj.data.vertex_colors:
                    obj.data.vertex_colors.remove(obj.data.vertex_colors[BLENDIT_BLEND_ID_NAME])

                if blendit_locals.blended_normals: 
                    bpy.context.view_layer.objects.active = obj 
                    bpy.ops.mesh.customdata_custom_splitnormals_clear()
                    blendit_locals.blended_normals = False
                    blendit_locals.blend_normals = False
                    
                if BLENDIT_ID_NAME in obj.data.uv_layers: 
                    bpy.context.view_layer.objects.active = obj 
                    obj.data.uv_layers[BLENDIT_ID_NAME].active = True
                    bpy.ops.mesh.uv_texture_remove()

                blendit_locals.blended = False

        bpy.context.view_layer.objects.active = active_obj 
        return {'FINISHED'}

class Blendit_OT_Switch_Mode(bpy.types.Operator):
    bl_label = "Switch Mode"
    bl_idname = "blendit.switch_mode"
    bl_description = "Switch between Simple and Advanced blend modes"
    bl_options = {'INTERNAL'}

    advanced: BoolProperty(
        name = "",
        default = False
        )

    @classmethod
    def poll(cls, context):
        scene = context.scene
        p = (context.mode == 'OBJECT' and
            isinstance(context.active_object, bpy.types.Object) and
            isinstance(context.active_object.data, bpy.types.Mesh))
        return p

    def execute(self, context):

        def reset_preview_mode(_obj): 
            try: 
                mix_node = _obj.material_slots[0].material.node_tree.nodes[BLENDIT_NODEGROUP_MIX]
            except:
                mix_node = None
            if mix_node:
                if mix_node.mute:
                    bpy.ops.blendit.toggle_preview('INVOKE_DEFAULT', toggle_mode = 0)
            return

        obj = context.object
        blendit_locals = obj.blendit_locals
        
        if self.advanced:
            if not blendit_locals.advanced_mod:
                blendit_locals.advanced_mod = True
                reset_preview_mode(obj)
                bpy.ops.blendit.blendit('INVOKE_DEFAULT', action = "switch")
        else:
            if blendit_locals.advanced_mod:
                blendit_locals.advanced_mod = False
                reset_preview_mode(obj)
                bpy.ops.blendit.blendit('INVOKE_DEFAULT', action = "switch")
 
        return {'FINISHED'}


class Blendit_OT_Switch_Style(bpy.types.Operator):
    bl_label = "Switch Style"
    bl_idname = "blendit.switch_style"
    bl_description = "Switch style of blend gradient"
    bl_options = {'INTERNAL'}

    style: IntProperty(
        name = "",
        default = 0
        )

    @classmethod
    def poll(cls, context):
        scene = context.scene
        p = (context.mode == 'OBJECT' and
            isinstance(context.active_object, bpy.types.Object) and
            isinstance(context.active_object.data, bpy.types.Mesh))
        return p

    def execute(self, context):

        def reset_preview_mode(_obj): 
            try: 
                mix_node = _obj.material_slots[0].material.node_tree.nodes[BLENDIT_NODEGROUP_MIX]
            except:
                mix_node = None
            if mix_node:
                if mix_node.mute:
                    bpy.ops.blendit.toggle_preview('INVOKE_DEFAULT', toggle_mode = 0)
            return

        obj = context.object
        blendit_locals = obj.blendit_locals
        
        if blendit_locals.blend_style != self.style:
            blendit_locals.blend_style = self.style
            reset_preview_mode(obj)
            bpy.ops.blendit.blendit('INVOKE_DEFAULT', action = "switch")
 
        return {'FINISHED'}


class Blendit_OT_Toggle_Preview(bpy.types.Operator):
    bl_label = "Toggle Preview Mode"
    bl_idname = "blendit.toggle_preview"
    bl_description = "Toggle Blend Gradient Preview"
    bl_options = {'INTERNAL'}

    toggle_mode: IntProperty(
        name = "",
        default = 0
        )

    @classmethod
    def description(cls, context, properties):
        if properties.toggle_mode == 0:
            tooltip = "Show / Hide Blend"
        elif properties.toggle_mode == 1:
            tooltip = "Show Blend Gradient"
        elif properties.toggle_mode == 2:
            tooltip = "Show Blend Gradient with Thresholds"
        return tooltip 

    @classmethod
    def poll(cls, context):
        scene = context.scene
        p = (context.mode == 'OBJECT' and
            isinstance(context.active_object, bpy.types.Object) and
            isinstance(context.active_object.data, bpy.types.Mesh))
        return p

    def execute(self, context):

        def switch_objects_modifiers(): 
            for _obj in context.scene.objects: 
                if _obj.blendit_locals.blended and _obj.material_slots[0].material == material:

                    _mod_name = BLENDIT_ID_NAME + "_V_W_Proximity"
                    if _mod_name in _obj.modifiers:
                        _obj.modifiers[_mod_name].show_viewport = not mix_node.mute

                    _mod_name = BLENDIT_ID_NAME + "_D_Transfer_Vcol"
                    if _mod_name in _obj.modifiers:
                        _obj.modifiers[_mod_name].show_viewport = not mix_node.mute
                    
                    _mod_name = BLENDIT_ID_NAME + "_D_Transfer_Norm"
                    if _mod_name in _obj.modifiers:
                        _obj.modifiers[_mod_name].show_viewport = not mix_node.mute

                    _mod_name = BLENDIT_ID_NAME + "_D_Transfer_UV"
                    if _mod_name in _obj.modifiers:
                        _obj.modifiers[_mod_name].show_viewport = not mix_node.mute

                    _mod_name = BLENDIT_ID_NAME + "_UV_Warp"
                    if _mod_name in _obj.modifiers:
                        _obj.modifiers[_mod_name].show_viewport = not mix_node.mute
            return

        obj = context.object
        blendit_locals = obj.blendit_locals

        try: 
            mix_node = obj.material_slots[0].material.node_tree.nodes[BLENDIT_NODEGROUP_MIX]
            material = obj.material_slots[0].material
        except:
            mix_node = None
            material = None

        if mix_node:
            if self.toggle_mode == 0:
                if mix_node.outputs[1].links: 
                    link = mix_node.outputs[1].links[0]
                    output_node = obj.material_slots[0].material.node_tree.nodes[link.to_node.name]
                    obj.material_slots[0].material.node_tree.links.remove(link)
                    obj.material_slots[0].material.node_tree.links.new(mix_node.outputs[0], output_node.inputs[0])

                elif mix_node.outputs[2].links: 
                    link = mix_node.outputs[2].links[0]
                    output_node = obj.material_slots[0].material.node_tree.nodes[link.to_node.name]
                    obj.material_slots[0].material.node_tree.links.remove(link)
                    obj.material_slots[0].material.node_tree.links.new(mix_node.outputs[0], output_node.inputs[0])

                mix_node.mute = not mix_node.mute
                switch_objects_modifiers()

            elif self.toggle_mode == 1:
                if mix_node.mute:
                    mix_node.mute = False
                    switch_objects_modifiers()

                if mix_node.outputs[0].links:
                    link = mix_node.outputs[0].links[0]
                    output_node = obj.material_slots[0].material.node_tree.nodes[link.to_node.name]
                    obj.material_slots[0].material.node_tree.links.remove(link)
                    obj.material_slots[0].material.node_tree.links.new(mix_node.outputs[1], output_node.inputs[0])

                elif mix_node.outputs[1].links:
                    link = mix_node.outputs[1].links[0]
                    output_node = obj.material_slots[0].material.node_tree.nodes[link.to_node.name]
                    obj.material_slots[0].material.node_tree.links.remove(link)
                    obj.material_slots[0].material.node_tree.links.new(mix_node.outputs[0], output_node.inputs[0])

                elif mix_node.outputs[2].links:
                    link = mix_node.outputs[2].links[0]
                    output_node = obj.material_slots[0].material.node_tree.nodes[link.to_node.name]
                    obj.material_slots[0].material.node_tree.links.remove(link)
                    obj.material_slots[0].material.node_tree.links.new(mix_node.outputs[1], output_node.inputs[0])

            elif self.toggle_mode == 2:
                if mix_node.mute:
                    mix_node.mute = False
                    switch_objects_modifiers()

                if mix_node.outputs[0].links:
                    link = mix_node.outputs[0].links[0]
                    output_node = obj.material_slots[0].material.node_tree.nodes[link.to_node.name]
                    obj.material_slots[0].material.node_tree.links.remove(link)
                    obj.material_slots[0].material.node_tree.links.new(mix_node.outputs[2], output_node.inputs[0])

                elif mix_node.outputs[1].links:
                    link = mix_node.outputs[1].links[0]
                    output_node = obj.material_slots[0].material.node_tree.nodes[link.to_node.name]
                    obj.material_slots[0].material.node_tree.links.remove(link)
                    obj.material_slots[0].material.node_tree.links.new(mix_node.outputs[2], output_node.inputs[0])

                elif mix_node.outputs[2].links:
                    link = mix_node.outputs[2].links[0]
                    output_node = obj.material_slots[0].material.node_tree.nodes[link.to_node.name]
                    obj.material_slots[0].material.node_tree.links.remove(link)
                    obj.material_slots[0].material.node_tree.links.new(mix_node.outputs[0], output_node.inputs[0])

        return {'FINISHED'}


class Blendit_OT_Relink(bpy.types.Operator):
    bl_label = "Relink Inputs"
    bl_idname = "blendit.relink"
    bl_description = "Reconnect the blend style input(s) with reroute node(s) by label"
    bl_options = {'INTERNAL'}

    @classmethod
    def poll(cls, context):
        scene = context.scene
        p = (context.mode == 'OBJECT' and
            isinstance(context.active_object, bpy.types.Object) and
            isinstance(context.active_object.data, bpy.types.Mesh))
        return p

    def execute(self, context):
        obj = context.object
        blendit_locals = obj.blendit_locals

        try: 
            _material = obj.material_slots[0].material
        except:
            _material = None

        bpy.ops.blendit.blendit('INVOKE_DEFAULT', action = "relink")

        return {'FINISHED'}


class Blendit_OT_Dialog_Warning(bpy.types.Operator):
    bl_idname = "blendit.dialog_warning"
    bl_label = "Blendit Warning"
    bl_options = {'INTERNAL'}

    warning: IntProperty(
        name = "",
        default = 0
        )

    def invoke(self, context, event):
        wm = context.window_manager
        wm.invoke_props_dialog(self, width = 300)
        return {'RUNNING_MODAL'} 
    
    def execute(self, context): 
        return {'FINISHED'}

    def draw(self, context):
        layout = self.layout
        
        show = self.warning & 0b001
        if show:
            layout.label(text="Objects must have specified materials.",icon='ERROR')

        show = self.warning & 0b010
        show = show >> 1
        if show:
            layout.label(text="The source material has an unsupported node setup.",icon='ERROR')

        show = self.warning & 0b100
        show = show >> 2
        if show:
            layout.label(text="The source material uses an unsupported mapping.",icon='ERROR')

        show = self.warning & 0b1000
        show = show >> 3
        if show:
            layout.label(text="The original material not found.",icon='ERROR')

        show = self.warning & 0b10000
        show = show >> 4
        if show:
            layout.label(text="Some hidden objects remain blended.",icon='ERROR')

        layout.separator()


class Blendit_PT_Panel:
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "Blendit"
    bl_context = "objectmode"

    @classmethod
    def poll(cls, context):
        p = context.mode == 'OBJECT' 
        
        return p


class Blendit_PT_Panel_Main(Blendit_PT_Panel, bpy.types.Panel): 
    bl_label = bl_info['name']+' v'+str(bl_info['version'])[1:2]+'.'+str(bl_info['version'])[4:5]
    bl_idname = "OBJECT_PT_BLENDITADDON"

    def draw(self, context):
        scene = context.scene
        blendit_globals = scene.blendit_globals
        obj = context.object
        blendit_locals = obj.blendit_locals

        layout = self.layout
        layout.use_property_decorate = False

        layout.use_property_split = True 
        layout.label(text="Settings:")
        row = layout.box().row()
        if ((blendit_globals.transfer_uv and not blendit_globals.blend_normals) or
            (blendit_globals.blend_normals and not blendit_globals.transfer_uv and not blendit_globals.auto_smooth)):
            col = row.column()
            col.ui_units_x = 0.1
            col.active = False
            col.label(text="")
            col.label(text="",icon='ERROR')

        if bpy.context.engine == "CYCLES" and blendit_globals.auto_smooth and blendit_globals.blend_normals:
            if ((blendit_globals.transfer_uv and not blendit_globals.blend_normals) or
                (blendit_globals.blend_normals and not blendit_globals.transfer_uv and not blendit_globals.auto_smooth)):
                col.label(text="")
                col.label(text="",icon='ERROR')
            else:
                col = row.column()
                col.ui_units_x = 0.1
                col.active = False
                col.label(text="")
                col.label(text="")
                col.label(text="")
                col.label(text="",icon='ERROR')

        col = row.column()
        col.prop(blendit_globals, "new_mat")
        col.prop(blendit_globals, "blend_normals")
        col.prop(blendit_globals, "transfer_uv")
        row3 = col.row()
        row3.prop(blendit_globals, "auto_smooth")
        row3.active = blendit_globals.blend_normals

        layout.use_property_split = True 
        layout.label(text="New Blend From:")
        row = layout.box().row()
        col = row.column(align=True)
        col.alignment = "LEFT"
        col.label(text="Blend Source:")
        col.label(text="")
        col.label(text="For Blending:")
        col = row.column(align=True)
        col.alignment = "RIGHT"
        col.label(text=context.object.name)
        if blendit_locals.blended:
            if blendit_locals.applied:
                col.label(text="[BLEND APPLIED]")
            else:
                col.label(text="[BLENDED]")
        elif blendit_locals.blend_source:
            col.label(text="[SOURCE]")
        else:
            col.label(text="")
        _count = len(context.selected_objects)-1*int(obj.select_get())
        col.label(text=str(_count))

        layout.use_property_split = False 
        row = layout.box().row()
        col = row.column(align=True)
        row2 = col.row()
        row2.operator("blendit.blendit", icon='OUTLINER_OB_META')
        row2.active = 1 < len(context.selected_objects) and not obj.blendit_locals.blended

        col.separator()
        col.operator("blendit.apply", icon='IMPORT')

        col.operator("blendit.remove", icon='TRASH')

        if isinstance(context.active_object, bpy.types.Object) and isinstance(context.active_object.data, bpy.types.Mesh):
            layout.use_property_split = False 
            layout.label(text="Blend:")
            
            row = layout.box().row()
            row.active = blendit_locals.blended
            if blendit_locals.advanced_mod:
                row.operator("blendit.switch_mode", text="Simple", emboss=False).advanced = False
                row.operator("blendit.switch_mode", text="Advanced", emboss=True).advanced = True
            else:
                row.operator("blendit.switch_mode", text="Simple", emboss=True).advanced = False
                row.operator("blendit.switch_mode", text="Advanced", emboss=False).advanced = True
            
            if not blendit_locals.applied or blendit_locals.advanced_mod:
                row = layout.box().row()
                col = row.column(align=True)
                row2 = col.row()
            else:
                row = layout.row()
                col = row.column(align=True)
                row2 = col.row()

            if not blendit_locals.applied:
                if blendit_locals.blend_source or not blendit_locals.blended:
                    if blendit_locals.advanced_mod:
                        col.prop(blendit_locals, "blend_min_dist", slider=True)
                        
                        col.prop(blendit_locals, "blend_dist", slider=True)
                        col.active = blendit_locals.blend_source
                    else:
                        row2.prop(blendit_locals, "blend_dist", slider=True)
                        row2.active = blendit_locals.blend_source
                else: 
                    if blendit_locals.advanced_mod:
                        col.prop(blendit_locals.source_obj.blendit_locals, "blend_min_dist", slider=True)
                        
                        col.prop(blendit_locals.source_obj.blendit_locals, "blend_dist", slider=True)
                    else:
                        row2.prop(blendit_locals.source_obj.blendit_locals, "blend_dist", slider=True)

                if blendit_locals.blended:
                    col.separator()
                    row2 = col.row()
                    row2.prop(blendit_locals, "blend_scale", slider=True)
                    row2.active = blendit_locals.blended

            if blendit_locals.advanced_mod and blendit_locals.blended:
                row2 = col.row()
                col = row2.column()
                if not blendit_locals.applied:
                    col.separator()
                col.label(text="Tweak Material Blend:")
                try:
                    cr_node = obj.material_slots[0].material.node_tree.nodes[BLENDIT_ID_NAME + 'ColorRamp']
                    col.template_color_ramp(cr_node, "color_ramp", expand=True)
                except:
                    col.label(text="Important Blendit nodes not found.",icon='ERROR')
                
                if blendit_locals.blend_normals:
                    col.separator()
                    col.label(text="Normals:")
                    row2 = col.row()
                    row2.prop(blendit_locals, "blend_normal_factor", text="Blend Factor", slider=True)

            layout.use_property_split = False 
            layout.label(text="Blend Style:")
            
            row = layout.box().row()
            row.active = blendit_locals.blended
            if blendit_locals.blend_style == 0:
                row.operator("blendit.switch_style", text="None", emboss=True).style = 0
            else:
                row.operator("blendit.switch_style", text="None", emboss=False).style = 0

            if blendit_locals.blend_style == 1:
                row.operator("blendit.switch_style", text="Noise", emboss=True).style = 1
            else:
                row.operator("blendit.switch_style", text="Noise", emboss=False).style = 1

            if blendit_locals.blend_style == 2:
                row.operator("blendit.switch_style", text="Heights", emboss=True).style = 2
            else:
                row.operator("blendit.switch_style", text="Heights", emboss=False).style = 2

            if blendit_locals.blend_style == 3:
                row.operator("blendit.switch_style", text="Custom", emboss=True).style = 3
            else:
                row.operator("blendit.switch_style", text="Custom", emboss=False).style = 3

            col = layout.box().column() 
            row = col.row()
            if blendit_locals.blended:
                try: 
                    mix_node = obj.material_slots[0].material.node_tree.nodes[BLENDIT_NODEGROUP_MIX]
                except:
                    mix_node = None
                if mix_node:
                    row.label(text="Preview Modes:")
                    row.alignment = "RIGHT"

                    if mix_node.mute:
                        row.operator("blendit.toggle_preview", text="", emboss=True, depress=True, icon='HIDE_ON').toggle_mode = 0
                        row.operator("blendit.toggle_preview", text="", emboss=False, icon='IMAGE_ALPHA').toggle_mode = 1
                        row.operator("blendit.toggle_preview", text="", emboss=False, icon='NODE_TEXTURE').toggle_mode = 2
                    else:
                        row.operator("blendit.toggle_preview", text="", emboss=False, icon='HIDE_OFF').toggle_mode = 0
                        if mix_node.outputs[1].links:
                            row.operator("blendit.toggle_preview", text="", emboss=True, depress=True, icon='IMAGE_ALPHA').toggle_mode = 1
                        else:
                            row.operator("blendit.toggle_preview", text="", emboss=False, icon='IMAGE_ALPHA').toggle_mode = 1
                        if mix_node.outputs[2].links:
                            row.operator("blendit.toggle_preview", text="", emboss=True, depress=True, icon='NODE_TEXTURE').toggle_mode = 2
                        else:
                            row.operator("blendit.toggle_preview", text="", emboss=False, icon='NODE_TEXTURE').toggle_mode = 2

                if blendit_locals.blend_style == 1: 
                    try:
                        style_node = obj.material_slots[0].material.node_tree.nodes[BLENDIT_ID_NAME + '_Noise_Blend']
                        col.separator()
                        col.prop(style_node.inputs[1], "default_value", text="Seed", slider=True)
                        col.separator()
                        col.prop(style_node.inputs[2], "default_value", text="Amplitude", slider=True)
                        col.prop(style_node.inputs[3], "default_value", text="Threshold", slider=True)
                        col.separator()
                        col.prop(style_node.inputs[4], "default_value", text="Frequency", slider=True)
                        col.prop(style_node.inputs[5], "default_value", text="Detail", slider=True)
                        col.prop(style_node.inputs[6], "default_value", text="Roughness", slider=True)
                        col.prop(style_node.inputs[7], "default_value", text="Distortion", slider=True)
                        col.separator()
                        col.prop(style_node.inputs[8], "default_value", text="Radius", slider=True)
                    except:
                        col.label(text="Important Blendit nodes not found.",icon='ERROR')

                if blendit_locals.blend_style == 2: 
                    try:
                        blendit_material = obj.material_slots[0].material.node_tree.blendit_material

                        style_node = obj.material_slots[0].material.node_tree.nodes[BLENDIT_ID_NAME + '_Heights_Blend']
                        col.separator()
                        row = col.row()
                        row.use_property_split = True
                        if style_node.inputs[1].is_linked and style_node.inputs[4].is_linked: 
                            row.prop(blendit_material, "style_reroute")
                        else:
                            row.prop(blendit_material, "style_reroute", icon='ERROR')

                        if blendit_locals.blended:
                            row.operator("blendit.relink", text="", emboss=False, icon='FILE_REFRESH')
                        
                        col.label(text="Object Heights:")
                        col.prop(style_node.inputs[2], "default_value", text="Scale", slider=True)
                        col.prop(style_node.inputs[3], "default_value", text="Threshold", slider=True)
                        
                        col.label(text="Source Heights:")
                        col.prop(style_node.inputs[5], "default_value", text="Scale", slider=True)
                        col.prop(style_node.inputs[6], "default_value", text="Threshold", slider=True)
                        col.separator()
                        col.prop(style_node.inputs[7], "default_value", text="Radius", slider=True)
                    except:
                        col.label(text="Important Blendit nodes not found.",icon='ERROR')

                if blendit_locals.blend_style == 3: 
                    try:
                        blendit_material = obj.material_slots[0].material.node_tree.blendit_material

                        style_node = obj.material_slots[0].material.node_tree.nodes[BLENDIT_ID_NAME + '_Custom_Blend']
                        col.separator()
                        row = col.row()
                        row.use_property_split = True
                        if style_node.inputs[1].is_linked:  
                            row.prop(blendit_material, "style_reroute")
                        else:
                            row.prop(blendit_material, "style_reroute", icon='ERROR')

                        if blendit_locals.blended:
                            row.operator("blendit.relink", text="", emboss=False, icon='FILE_REFRESH')
                        col.separator()
                        col.prop(style_node.inputs[2], "default_value", text="Amplitude", slider=True)
                        col.prop(style_node.inputs[3], "default_value", text="Threshold", slider=True)
                        col.separator()
                        col.prop(style_node.inputs[4], "default_value", text="Radius", slider=True)
                    except:
                        col.label(text="Important Blendit nodes not found.",icon='ERROR')

                if blendit_locals.reused_uv: 
                    layout.use_property_split = False
                    layout.label(text="UV Mapping:")
                    col = layout.box().column()
                    row2 = col.row()
                    row2.prop(blendit_locals, "uv_scale", slider=True)

            else:
                
                row.label(text="New Blend Defaults:")
                col.active = False

                if blendit_locals.blend_style == 1: 
                    col.separator()
                    col.prop(blendit_locals, "style_seed", slider=True)
                    col.separator()
                    col.prop(blendit_locals, "style_amplitude", slider=True)
                    col.prop(blendit_locals, "style_brightness", slider=True)
                    col.separator()
                    col.prop(blendit_locals, "style_frequency", slider=True)
                    col.prop(blendit_locals, "style_detail", slider=True)
                    col.prop(blendit_locals, "style_roughness", slider=True)
                    col.prop(blendit_locals, "style_distortion", slider=True)
                    col.separator()
                    col.prop(blendit_locals, "style_smooth", slider=True)

                if blendit_locals.blend_style == 2: 
                    col.separator()
                    row = col.row()
                    row.use_property_split = True
                    row.prop(blendit_locals, "style_reroute")
                    if blendit_locals.blended:
                        row.operator("blendit.relink", text="", emboss=False, icon='FILE_REFRESH')
                    col.separator()
                    col.prop(blendit_locals, "style_scale", slider=True)
                    col.prop(blendit_locals, "style_brightness", slider=True)
                    col.separator()
                    col.prop(blendit_locals, "style_scale2", slider=True)
                    col.prop(blendit_locals, "style_brightness2", slider=True)
                    col.separator()
                    col.prop(blendit_locals, "style_smooth", slider=True)

                if blendit_locals.blend_style == 3: 
                    col.separator()
                    row = col.row()
                    row.use_property_split = True
                    row.prop(blendit_locals, "style_reroute")
                    if blendit_locals.blended:
                        row.operator("blendit.relink", text="", emboss=False, icon='FILE_REFRESH')
                    col.separator()
                    col.prop(blendit_locals, "style_amplitude", slider=True)
                    col.prop(blendit_locals, "style_brightness", slider=True)
                    col.separator()
                    col.prop(blendit_locals, "style_smooth", slider=True)


classes = (
    BlenditSceneProperties,
    BlenditMaterialProperties,
    BlenditObjectProperties,
    Blendit_OT_Blendit,
    Blendit_OT_Apply,
    Blendit_OT_Remove,
    Blendit_OT_Dialog_Warning,
    Blendit_OT_Switch_Mode,
    Blendit_OT_Switch_Style,
    Blendit_OT_Toggle_Preview,
    Blendit_OT_Relink,
    Blendit_PT_Panel_Main
    )


def register():
    for cls in classes:
        register_class(cls)

    bpy.types.Scene.blendit_globals = PointerProperty(type=BlenditSceneProperties)
    bpy.types.NodeTree.blendit_material = PointerProperty(type=BlenditMaterialProperties)
    bpy.types.Object.blendit_locals = PointerProperty(type=BlenditObjectProperties)

 
def unregister():

    for cls in reversed(classes):
        unregister_class(cls)

    del bpy.types.Scene.blendit_globals
    del bpy.types.NodeTree.blendit_material
    del bpy.types.Object.blendit_locals
 
if __name__ == "__main__":
    register()
